import logging
import numpy as np
import uuid
from rest_framework import exceptions as drf_exceptions
from yaphone.advisor.common import exceptions

from yaphone.advisor.common.tools import rewrite_params_in_url
from yaphone.advisor.launcher.models import wallpapers as wallpapers_models

logger = logging.getLogger(__name__)

WALLPAPER_COVER_HEIGHT_DP = 130

PREVIEW_SMALL = 'small'
PREVIEW_MEDIUM = 'medium'
PREVIEW_LARGE = 'large'
PREVIEW_XLARGE = 'xlarge'

PREVIEW_IMAGE_SIZES_DP = {
    PREVIEW_SMALL: 16,
    PREVIEW_MEDIUM: 88,
    PREVIEW_LARGE: 192,
    PREVIEW_XLARGE: 270,
}

DEFAULT_PREVIEW_IMAGE_SIZES = (PREVIEW_SMALL, PREVIEW_MEDIUM, PREVIEW_LARGE)
DEFAULT_FEED_STRATEGY = 'most_picked'


def normalize_image_key(key):
    return key.split('/')[-1]


def get_image_key(image):
    return normalize_image_key(image.image.key)


class BaseWallpapersIssuer(object):
    """
    Base class for issuing wallpapers in different order
    Order is controlled by overloading get_ratings_dict and optionally get_rating
    """

    def __init__(self):
        """
        In init the object gets ratings for all wallpapers
        by which they will be sorted in decreasing order
        It needs to be overloaded to provide
        self.ratings_dict: dict of type {image_key, rating}
        """
        self.ratings_dict = None

    def get_rating(self, image_key):
        """
        Get rating from self.ratings_dict for specific image
        Can be overloaded to override default rating in some tricky way
        :return: rating of wallpaper
        """
        return self.ratings_dict.get(image_key, 0)

    def get_wallpapers(self, categories, blacklist=None):
        """
        Get wallpapers list from categories and sorts it by ratings
        :param categories: iterable of WallpaperCategories
        :param blacklist: iterable of image keys
        :return: list of WallpaperImages
        """
        blacklist = blacklist or []
        wallpapers = []
        for category in categories:
            for image in category.images:
                image_key = get_image_key(image)
                if image_key not in blacklist:
                    rating = self.get_rating(image_key)
                    wallpaper = {'image': image,
                                 'category': category,
                                 'rating': rating}
                    wallpapers.append(wallpaper)
        wallpapers.sort(key=lambda wp: wp['rating'], reverse=True)
        return wallpapers


class RandomWallpapersIssuer(BaseWallpapersIssuer):
    def __init__(self):
        self.ratings_dict = {}

    def get_rating(self, image_key):
        return np.random.random()


class MostPickedWallpapersIssuer(BaseWallpapersIssuer):
    def __init__(self):
        wallpapers_pick_counts = \
            wallpapers_models.WallpaperStatus.objects().scalar('wallpaper', 'collection_pick_count')
        self.ratings_dict = dict(wallpapers_pick_counts)


class LessSkippedWallpapersIssuer(BaseWallpapersIssuer):
    def __init__(self):
        wallpapers_auto_next_rates = \
            wallpapers_models.WallpaperStatus.objects().scalar('wallpaper', 'auto_next_rate')
        self.ratings_dict = dict(wallpapers_auto_next_rates)


class BanditsWallpapersIssuer(BaseWallpapersIssuer):
    def __init__(self):
        wallpapers, wins, fails = list(zip(
            *wallpapers_models.WallpaperStatus.objects().scalar('wallpaper', 'auto_next_count', 'force_next_count')
        ))
        thetas = self.get_thetas(wins, fails)
        self.ratings_dict = dict(zip(wallpapers, thetas))

    @staticmethod
    def get_thetas(wins, fails):
        """
        Returns theta for Thomspson sampling (h.yandex-team.ru/?bit.ly/2JbvEqw)
        :param wins: wins (positive feedback) for every wallpaper
        :param fails: fails (negative feedback) for every wallpaper
        :return: an array of floats - samples from beta distribution
        """
        thetas = np.random.beta(np.fromiter((win or 0 for win in wins), dtype=np.int) + 1,
                                np.fromiter((fail or 0 for fail in fails), dtype=np.int) + 1)
        return thetas

    def get_rating(self, image_key):
        if image_key in self.ratings_dict:
            return self.ratings_dict[image_key]
        else:
            return np.random.random()


def get_wallpapers_feed(feed_id, color=None, offset=None, limit=None, no_dereference=True):
    if offset is not None:
        wallpapers_feed = wallpapers_models.WallpapersFeed.objects(feed_id=feed_id, color=color) \
            .fields(slice__wallpapers=(offset, limit))
    else:
        wallpapers_feed = wallpapers_models.WallpapersFeed.objects(feed_id=feed_id, color=color)
    if no_dereference:
        wallpapers_feed = wallpapers_feed.no_dereference()
    return wallpapers_feed.first()


class BaseWallpapersFeedLoader(object):
    def __init__(self, color_filters):
        self.color_filters = color_filters

    @staticmethod
    def dereference_badges_and_colors(wallpapers, color_filters):
        badges = wallpapers_models.WallpaperBadges.objects()
        colors_dict = {color.id: color for color in color_filters}
        badges_dict = {badge.id: badge for badge in badges}
        for wallpaper in wallpapers:
            wallpaper.colors = [colors_dict[color.id] for color in wallpaper.colors]
            wallpaper.badges = [badges_dict[badge.id] for badge in wallpaper.badges]
        return wallpapers

    def load_feed(self, feed_id, offset, limit):
        wallpapers = self._load_feed(feed_id, offset, limit)
        wallpapers = self.dereference_badges_and_colors(wallpapers, self.color_filters)
        return wallpapers

    def _load_feed(self, feed_id, offset, limit):
        raise NotImplementedError


class NewWallpapersFeedLoader(BaseWallpapersFeedLoader):
    def __init__(self, color_filters, uuid_, categories, strategy):
        super(NewWallpapersFeedLoader, self).__init__(color_filters)
        self.uuid_ = uuid_
        self.categories = categories
        self.strategy = strategy

    @staticmethod
    def shuffle_wallpapers(wallpapers, sigma=5):
        """
        Shuffles wallpapers lightly: generates new index for each wallpaper
        from normal distribution with mean in current index and given sigma
        :param wallpapers: array of wallpapers
        :param sigma: float - standard deviation of normal distribution
        controls the intensity of shuffling.
        :return: shuffled array of wallpapers
        """
        ranks = np.random.normal(range(len(wallpapers)), sigma)
        return [wallpaper for rank, wallpaper in sorted(zip(ranks, wallpapers))]

    def _load_feed(self, feed_id, offset, limit):
        logger.debug('New feed. feed_id: "%s"', feed_id)
        wallpapers = []
        for wallpaper in wallpapers_issuers[self.strategy]().get_wallpapers(self.categories):
            image = wallpaper['image']
            image.collection_id = wallpaper['category'].id_
            wallpapers.append(image)
        if not wallpapers:
            logger.error('Constructed empty wallpapers feed: %s', feed_id)
        wallpapers = self.shuffle_wallpapers(wallpapers)
        wallpapers_feed = wallpapers_models.WallpapersFeed(uuid=self.uuid_, wallpapers=wallpapers,
                                                           feed_id=feed_id, color=None)
        wallpapers_feed.save()
        wallpapers = wallpapers[offset:offset + limit]
        return wallpapers


class OldWallpapersFeedLoader(BaseWallpapersFeedLoader):
    def _load_feed(self, feed_id, offset, limit):
        wallpapers_feed = get_wallpapers_feed(feed_id, offset=offset, limit=limit)
        if not wallpapers_feed:
            logger.warning('Feed with id "%s" not found', feed_id)
            raise drf_exceptions.NotFound('No feed with such id found')
        logger.debug('Old feed. feed_id: "%s"', feed_id)
        wallpapers = wallpapers_feed.wallpapers
        return wallpapers


class OldColoredWallpapersFeedLoader(BaseWallpapersFeedLoader):
    def __init__(self, color_filters, wallpapers_feed):
        super(OldColoredWallpapersFeedLoader, self).__init__(color_filters)
        self.wallpapers_feed = wallpapers_feed

    def _load_feed(self, feed_id, offset, limit):
        logger.debug('Old colored feed. feed_id: "%s" color: "%s"', feed_id, self.wallpapers_feed.color)
        wallpapers = self.wallpapers_feed.wallpapers
        return wallpapers


class NewColoredWallpapersFeedLoader(BaseWallpapersFeedLoader):
    def __init__(self, color_filters, uuid_, color_obj):
        super(NewColoredWallpapersFeedLoader, self).__init__(color_filters)
        self.uuid_ = uuid_
        self.color_obj = color_obj

    def _load_feed(self, feed_id, offset, limit):
        wallpapers_feed = get_wallpapers_feed(feed_id)
        if not wallpapers_feed:
            logger.warning('Feed with id "%s" not found', feed_id)
            raise drf_exceptions.NotFound('No feed with such feed_id found')

        logger.debug('New colored feed. feed_id: "%s" color: "%s"', feed_id, self.color_obj)
        wallpapers = [
            wallpaper for wallpaper in wallpapers_feed.wallpapers
            if self.color_obj in wallpaper.colors
        ]
        if not wallpapers:
            logger.error('Constructed empty colored wallpapers feed: %s', feed_id)
        wallpapers_models.WallpapersFeed(uuid=self.uuid_, wallpapers=wallpapers,
                                         feed_id=feed_id, color=self.color_obj).save()
        wallpapers = wallpapers[offset:offset + limit]
        return wallpapers


class WallpapersFeedBuilder(object):
    def __init__(self, request, uuid_, categories, color_group):
        self.request = request
        self.uuid_ = uuid_
        self.categories = categories
        self.color_group = color_group

    @staticmethod
    def make_next_url(request, feed_id, offset, limit):
        current_url = request.build_absolute_uri()
        return rewrite_params_in_url(current_url, {'offset': offset + limit, 'feed_id': feed_id})

    @classmethod
    def get_color_filters(cls, color_group):
        colors = wallpapers_models.WallpaperColors.objects(group=color_group).order_by('order')
        return colors

    @staticmethod
    def get_color_object(color):
        color_group, color_id = color.split('/')
        try:
            color_obj = wallpapers_models.WallpaperColors.objects.get(group=color_group, id_=color_id)
        except wallpapers_models.WallpaperColors.DoesNotExist:
            raise exceptions.BadRequestAPIError('Color "{}" does not exist'.format(color))
        return color_obj

    def load_wallpapers_feed(self, color_filters, feed_id, color, offset, limit, strategy=DEFAULT_FEED_STRATEGY):
        if not feed_id:
            feed_id = uuid.uuid4()
            loader = NewWallpapersFeedLoader(color_filters, self.uuid_, self.categories, strategy)
        elif not color:
            loader = OldWallpapersFeedLoader(color_filters)
        else:  # Color and feed_id are present
            color_obj = self.get_color_object(color)
            wallpapers_feed_colored = get_wallpapers_feed(feed_id, color=color_obj, offset=offset, limit=limit)
            if wallpapers_feed_colored:
                loader = OldColoredWallpapersFeedLoader(color_filters, wallpapers_feed_colored)
            else:
                loader = NewColoredWallpapersFeedLoader(color_filters, self.uuid_, color_obj)

        wallpapers = loader.load_feed(feed_id, offset, limit)
        return feed_id, wallpapers

    def build_feed(self, feed_id, color, offset, limit):
        color_filters = self.get_color_filters(self.color_group)
        feed_id, wallpapers = self.load_wallpapers_feed(color_filters=color_filters,
                                                        feed_id=feed_id, color=color,
                                                        offset=offset, limit=limit)
        next_url = self.make_next_url(request=self.request, feed_id=feed_id, offset=offset, limit=limit)
        feed = dict(
            lifetime_seconds=60 * 60 * 24,
            color_filters=color_filters,
            wallpapers=wallpapers,
            feed_id=feed_id,
            next_url=next_url,
        )
        return feed


wallpapers_issuers = {
    'bandits': BanditsWallpapersIssuer,
    'random': RandomWallpapersIssuer,
    'most_picked': MostPickedWallpapersIssuer,
    'less_skipped': LessSkippedWallpapersIssuer,
}
