import mongoengine as me
from datetime import datetime

from yaphone.advisor.advisor.impression_id import ImpressionID
from yaphone.advisor.advisor.models.profile import Profile
from yaphone.advisor.common.mongo_utils import ChangeLoggingMixin

GIFTS_ID_TITLE = 'gifts_yandex_title'
GIFTS_ERROR_TITLE = 'gifts_error_title'
GIFTS_ERROR_MESSAGE = 'gifts_error_message'
GIFTS_WRONG_ACCOUNT_TITLE = 'gifts_wrong_account_title'
GIFTS_WRONG_ACCOUNT_MESSAGE = 'gifts_wrong_account_message'
TITLE_ID_TEMPLATE = 'gift_yandex_{id}_title'
DESCRIPTION_ID_TEMPLATE = 'gift_yandex_{id}_description'
DISCLAIMER_ID_TEMPLATE = 'gift_yandex_{id}_disclaimer'
APP_NAME_ID_TEMPLATE = 'gift_yandex_{id}_feed_app_name'
FEED_TITLE_ID_TEMPLATE = 'gift_yandex_{id}_feed_title'
FEED_DESCRIPTION_ID_TEMPLATE = 'gift_yandex_{id}_feed_description'
DEFAULT_LANGUAGE = 'ru'
DEFAULT_DEVICE_NAME = 'Yandex YNDX-000SB'

DISK_ID = 'disk'
MONEY_ID = 'money'
TAXI_ID = 'taxi'
PLUS_ID = 'plus'

ALL_GIFTS = (DISK_ID, MONEY_ID, TAXI_ID, PLUS_ID)


# noinspection PyClassHasNoInit
class UpdateSubscriptions(me.Document):
    device_id = me.UUIDField(required=True)
    package_name = me.StringField(required=True)
    timestamp = me.DateTimeField(required=False, default=datetime.utcnow)

    meta = {
        'strict': False,
        'indexes': [
            {
                'fields': ('device_id', 'package_name')
            }
        ],
        'index_background': True,
        'auto_create_index': False,
    }


class LangSpecificValues(me.EmbeddedDocument):
    lang = me.StringField()
    title = me.StringField()
    description = me.StringField()

    def __unicode__(self):
        return u'%s: %s - %s' % (self.lang, self.title, self.description)


class Region(me.EmbeddedDocument):
    code = me.IntField(required=True)
    description = me.StringField(required=True)

    meta = {'strict': False}

    def __unicode__(self):
        return self.description


# noinspection PyClassHasNoInit
class YandexDistributedApp(me.Document):
    _impression_id = None

    class IncompleteDocument(Exception):
        pass

    package_name = me.StringField(required=True, unique=True)
    tag = me.StringField(required=False, choices=('popular', 'mobile', 'bank', 'dont_show'))
    tag_extra = me.DictField(required=False)
    texts = me.ListField(field=me.EmbeddedDocumentField(LangSpecificValues))
    offer_id = me.StringField(required=False)
    content_rating = me.StringField(choices=('3+', '7+', '12+', '16+', '18+'))
    download_url = me.URLField()
    icon = me.StringField(required=False)
    rank = me.IntField(required=False, default=0)
    mark_as_sponsored = me.BooleanField(required=False, default=False)
    regions = me.ListField(me.EmbeddedDocumentField(Region), required=False)

    def __get_lang_field(self, lang, field_name):
        for item in self.texts:
            if item.lang == lang:
                return getattr(item, field_name)

    def get_lang_field(self, lang, field_name):
        for item in (lang, DEFAULT_LANGUAGE):
            value = self.__get_lang_field(item, field_name)
            if value:
                return value

        raise self.IncompleteDocument(
            '%s not found for lang=%s, package_name=%s' % (field_name, lang, self.package_name))

    def get_title(self, lang):
        return self.get_lang_field(lang, 'title')

    def get_description(self, lang):
        return self.get_lang_field(lang, 'description')

    @property
    def is_promo(self):
        return bool(self.offer_id)

    meta = {
        'strict': False,
        'indexes': [
            'tag', 'package_name'
        ],
        'index_background': True,
        'auto_create_index': False,
    }

    @property
    def impression_id(self):
        if self._impression_id is None:
            self._impression_id = ImpressionID(content_type='apps')
        return self._impression_id


# noinspection PyClassHasNoInit
class Gifts(me.Document):
    id = me.StringField(primary_key=True)
    package_name = me.StringField(required=True, regex='^gift:[a-zA-Z0-9_.]+')
    preview_order = me.IntField(required=False, db_field='order')
    suw_preview_image = me.StringField(required=True)
    updated_at = me.DateTimeField(required=True, default=datetime.utcnow)

    meta = {
        'strict': False,
        'indexes': ['preview_order', 'package_name'],
        'index_background': True,
        'auto_create_index': False,
    }

    def __unicode__(self):
        return u'Gifts: {}'.format(self.pk)


class BonusCardText(me.EmbeddedDocument):
    lang = me.StringField(required=True)
    app_name = me.StringField(required=True)
    title = me.StringField(required=True)
    description = me.StringField(required=True)
    disclaimer = me.StringField(required=False, default=str)


class BonusCardResources(me.Document):
    package_name = me.StringField(primary_key=True)
    icon = me.StringField(required=True)
    banner_image = me.StringField(required=True)
    url_template = me.StringField(required=True)
    button_text_key = me.StringField(required=True)
    updated_at = me.DateTimeField(required=True, default=datetime.utcnow)
    texts = me.ListField(field=me.EmbeddedDocumentField(BonusCardText))
    mark_as_sponsored = me.BooleanField(required=False, default=False)
    content_rating = me.StringField(required=False, default=str)

    icon_colorwiz = me.DictField(required=False)
    banner_image_colorwiz = me.DictField(required=False)

    def get_text(self, language, field):
        texts = {text.lang: text for text in self.texts}
        text = texts.get(language) or texts.get(DEFAULT_LANGUAGE)
        if text:
            return getattr(text, field, None)

    meta = {
        'strict': False,
        'index_background': True,
        'auto_create_index': False,
    }

    def __unicode__(self):
        return self.id


class GivenGift(me.EmbeddedDocument):
    gift = me.LazyReferenceField(Gifts, required=True, passthrough=True, reverse_delete_rule=me.DO_NOTHING)
    activated = me.BooleanField(required=False, default=False)
    promocode = me.StringField(required=False)
    activated_at = me.DateTimeField(required=False)

    def __unicode__(self):
        return u'GivenGift: {}'.format(self.gift.pk)


class GiftSetError(Exception):
    pass


class GiftSet(ChangeLoggingMixin, me.Document):
    passport_uid = me.LongField(primary_key=True)
    gifts = me.ListField(me.EmbeddedDocumentField(GivenGift))
    created_at = me.DateTimeField(required=True, default=datetime.utcnow)

    def add_gifts(self, allowed_gifts=None):
        if allowed_gifts is None:
            allowed_gifts = ALL_GIFTS

        current_gifts = [item.gift.pk for item in self.gifts]
        self.gifts = [GivenGift(gift=gift) for gift in Gifts.objects.filter(pk__in=allowed_gifts,
                                                                            pk__not__in=current_gifts + [DISK_ID])]

    def get_given_gift_by_id(self, gift_id):
        for given_gift in self.gifts:
            if given_gift.gift.pk == gift_id:
                return given_gift
        else:
            raise GiftSetError("No GivenGift with id = {gift_id} found in {gift_set}".format(
                gift_id=gift_id, gift_set=self))

    def add_promocodes(self, promocodes):
        """
        Copy promocodes from promocodes dict
        :param promocodes: dict (key - Gift's id, value - promocode)
        Example
        {
            'taxi': 'taxi_promocode',
            'plus': 'plus_promocode'
        }
        """
        for item in self.gifts:
            gift_id = item.gift.pk
            if gift_id in promocodes:
                item.promocode = promocodes[gift_id]

    def activate_gift(self, gift_id):
        given_gift = self.get_given_gift_by_id(gift_id)
        if not given_gift.activated:
            given_gift.activated = True
            given_gift.activated_at = datetime.utcnow()

    def __unicode__(self):
        return u'GiftSet: {}'.format(self.passport_uid)

    meta = {
        'strict': False,
        'alias': 'primary_only',
    }


# noinspection PyClassHasNoInit
class Phone(ChangeLoggingMixin, me.Document):
    id = me.StringField(primary_key=True)  # aka phone_id
    # TODO: when protocol is finished and all views are added, make device_id required
    # temporary setting required=False for device_id
    device_id = me.UUIDField(required=False)  # TODO: update device_id if it changes
    activated = me.BooleanField(required=False, default=False)
    first_activation_datetime = me.DateTimeField(required=False)
    last_activation_datetime = me.DateTimeField(required=False)
    # currently logged-in user
    passport_uid = me.LongField(required=False)  # TODO: update if login with another uid
    # gifts info
    gift_set = me.LazyReferenceField(GiftSet, passthrough=True, reverse_delete_rule=me.NULLIFY)
    # each taxi promocode is strictly tied to specific Phone object
    taxi_promocode = me.StringField(required=False)
    # Some useful info about phones
    imei = me.StringField(required=False)
    imei2 = me.StringField(required=False)
    serial_number = me.StringField(required=False)
    wifi_mac = me.StringField(required=False)
    bluetooth_mac = me.StringField(required=False)
    batch = me.StringField(required=False)
    disk_autoupload = me.BooleanField(required=False, default=False)
    allowed_gifts = me.ListField(me.StringField(choices=ALL_GIFTS), required=False, default=ALL_GIFTS)

    meta = {
        'strict': False,
        'indexes': ['device_id', 'passport_uid', 'taxi_promocode'],
        'index_background': True,
        'auto_create_index': False,
        'alias': 'primary_only',
    }

    def __unicode__(self):
        return u'Phone: {}'.format(self.pk)

    def save(self, **kwargs):
        if self.device_id:
            Profile.objects(pk=self.device_id).update(upsert=True, phone_id=self.id)
        return super(Phone, self).save(**kwargs)


# noinspection PyClassHasNoInit
class PlusPromocode(ChangeLoggingMixin, me.Document):
    value = me.StringField(required=True)
    passport_uid = me.LongField(required=False)

    meta = {
        'strict': False,
        'indexes': ['passport_uid', 'value'],
        'index_background': True,
        'auto_create_index': False,
        'alias': 'primary_only',
    }

    def __unicode__(self):
        return u'Promocode {} (uid={})'.format(self.value, self.passport_uid)


PLUS_AUTO_SUBSCRIPTION = 'plus_auto_subscription'
PLUS_MANUAL_SUBSCRIPTION = 'plus_manual_subscription'
PLUS_PROMOCODE = 'plus_promocode'
NO_GIFTS = 'no_gifts'

PLUS_ONLY_PREFIX = 'plus_only_'


# noinspection PyClassHasNoInit
class GreetingMail(ChangeLoggingMixin, me.Document):
    passport_uid = me.LongField(required=True)
    plus_promocode = me.StringField(required=False)
    taxi_promocode = me.StringField(required=False)
    mail_type = me.StringField(
        required=True,
        choices=(
            PLUS_AUTO_SUBSCRIPTION,
            PLUS_MANUAL_SUBSCRIPTION,
            PLUS_PROMOCODE,
            NO_GIFTS,

            # https://st.yandex-team.ru/ADVISOR-2274
            PLUS_ONLY_PREFIX + PLUS_AUTO_SUBSCRIPTION,
            PLUS_ONLY_PREFIX + PLUS_MANUAL_SUBSCRIPTION,
            PLUS_ONLY_PREFIX + PLUS_PROMOCODE,
        )
    )
    phone_id = me.StringField(required=True)

    meta = {
        'strict': False,
        'indexes': ['passport_uid'],
        'index_background': True,
        'auto_create_index': False,
        'alias': 'primary_only',
    }

    def __unicode__(self):
        return u"Mail '{}' for uid={}".format(self.mail_type, self.passport_uid)
