import json
from datetime import datetime, timedelta
from time import time

import mock
from django.test import TestCase
from django.test.client import Client
from rest_framework import status

from yaphone.advisor.advisor.tests.fixtures import fake_collection
from yaphone.advisor.common.mocks.geobase import LookupMock
from yaphone.advisor.common.mocks.test_client_data import SOME_UUID, SOME_RUSSIAN_IP, SOME_DEVICE_ID
from yaphone.advisor.launcher.tests.base import StatelessViewMixin
from yaphone.advisor.setup_wizard.models import (Phone, GiftSet, PlusPromocode, GivenGift, Gifts, GreetingMail,
                                                 PLUS_ID, TAXI_ID, MONEY_ID, GiftSetError)
from yaphone.advisor.setup_wizard.views.gifts import GiftUsedView

SOME_PHONE_ID = 'some_phone_id'
ANOTHER_PHONE_ID = 'another_phone_id'
ANOTHER_DEVICE_ID = "22222222222222222222222222222222"
SOME_PASSPORT_UID = 77
SOME_PASSPORT_INFO = {
    'passport_uid': SOME_PASSPORT_UID,
    'has_plus': False,
    'is_portal_login': True,
}
HAS_PLUS_PASSPORT_INFO = {
    'passport_uid': SOME_PASSPORT_UID,
    'has_plus': True,
    'is_portal_login': True,
}
NON_PORTAL_PASSPORT_INFO = {
    'passport_uid': SOME_PASSPORT_UID,
    'has_plus': False,
    'is_portal_login': False,
}

NOT_RUSSIA_COUNTRY_ID = 206

TRANSLATION_KEYS = {
    'backend_jafar_setup_wizard_title': 'Recomendations Title',
    'backend_jafar_setup_wizard_subtitle': 'Recommendations Subtitle'
}


class GiftsMixin(object):
    def setUp(self):
        self.data = {
            'phone_id': SOME_PHONE_ID,
            'device_id': SOME_DEVICE_ID,
        }
        fake_collection('gifts.json', Gifts)
        super(GiftsMixin, self).setUp()

    def tearDown(self):
        Phone.objects.delete()
        GiftSet.objects.delete()
        Gifts.objects.delete()
        PlusPromocode.objects.delete()
        GreetingMail.objects.delete()

    @staticmethod
    def create_client(device_type=None):
        user_agent = 'com.yandex.phone.setup_wizard/1.0.qa2147483647 (Yandex Phone; Android 8.1)'
        if device_type:
            user_agent = '{} {}'.format(user_agent, device_type)
        return Client(
            HTTP_USER_AGENT=user_agent,
            HTTP_X_YAUUID=SOME_UUID,
            HTTP_HOST='setup-wizard.localhost',
            HTTP_X_YACLID1='2247990',
            HTTP_ACCEPT_LANGUAGE='ru_RU',
            HTTP_ACCEPT='application/json',
            HTTP_AUTHORIZATION='OAuth 6gcd927860e193603d4f7c9577cg8f5a',
            REMOTE_ADDR=SOME_RUSSIAN_IP,
        )

    def post(self, **kwargs):
        return self.client.post(self.endpoint, json.dumps(self.data), content_type="application/json", **kwargs)


@mock.patch('yaphone.utils.geo.geobase_lookuper', LookupMock())
class ActivationViewTest(GiftsMixin, StatelessViewMixin, TestCase):
    endpoint = '/api/v1/activation'

    def test_ok(self):
        Phone(pk=SOME_PHONE_ID).save()
        self.assertEqual(self.post().status_code, status.HTTP_200_OK)

    def test_timestamp(self):
        Phone(pk=SOME_PHONE_ID).save()
        response = json.loads(self.post().content)
        self.assertLess(time() - response['timestamp'], 1)

    def test_unknown_phone(self):
        self.assertEqual(self.post().status_code, status.HTTP_400_BAD_REQUEST)

    def test_phone_fields(self):
        phone = Phone(pk=SOME_PHONE_ID).save()
        self.assertFalse(phone.activated)
        utcnow = datetime.utcnow()
        self.post()
        phone = Phone.objects.get(pk=SOME_PHONE_ID)
        self.assertTrue(phone.activated)
        self.assertEqual(phone.device_id.hex, SOME_DEVICE_ID)
        self.assertLess(utcnow - phone.first_activation_datetime, timedelta(seconds=1))
        self.assertLess(utcnow - phone.last_activation_datetime, timedelta(seconds=1))


@mock.patch('yaphone.utils.geo.geobase_lookuper', LookupMock())
class GiveGiftsView(GiftsMixin, StatelessViewMixin, TestCase):
    endpoint = '/api/v1/give_gifts'

    @mock.patch('yaphone.advisor.common.passport.get_info', lambda *args, **kwargs: SOME_PASSPORT_INFO.copy())
    def test_ok(self):
        Phone(
            pk=SOME_PHONE_ID,
            device_id=SOME_DEVICE_ID
        ).save()
        response = self.post()
        self.assertEqual(response.status_code, status.HTTP_200_OK)

    @mock.patch('yaphone.advisor.common.passport.get_info', lambda *args, **kwargs: SOME_PASSPORT_INFO.copy())
    def test_give_gifts_user_has_no_plus(self):
        response = self.post()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertDictEqual(response.data, {'give_gifts': False, 'has_plus': False})

    @mock.patch('yaphone.advisor.common.passport.get_info', lambda *args, **kwargs: HAS_PLUS_PASSPORT_INFO.copy())
    def test_give_gifts_user_has_plus(self):
        response = self.post()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertDictEqual(response.data, {'give_gifts': False, 'has_plus': True})


@mock.patch('yaphone.utils.geo.geobase_lookuper', LookupMock())
class GiftsPreviewViewTest(GiftsMixin, StatelessViewMixin, TestCase):
    endpoint = '/api/v1/gifts_preview'

    def setUp(self):
        Phone(pk=SOME_PHONE_ID).save()
        super(GiftsPreviewViewTest, self).setUp()
        self.data = {'phone_id': SOME_PHONE_ID}

    def test_ok(self):
        self.assertEqual(self.post().status_code, status.HTTP_200_OK)

    def test_ok_get(self):
        self.assertEqual(self.get(params=self.data).status_code, status.HTTP_200_OK)

    def test_uuid_is_required(self, *args):
        self.assertEqual(self.post().status_code, status.HTTP_200_OK)

    def test_badly_formed_uuid(self, *args):
        self.assertEqual(self.post(HTTP_X_YAUUID='olololo').status_code, status.HTTP_200_OK)


@mock.patch('yaphone.utils.geo.geobase_lookuper', LookupMock())
class GiftUsedViewTest(GiftsMixin, StatelessViewMixin, TestCase):
    endpoint = '/api/v1/gift_used'

    @mock.patch('yaphone.advisor.common.passport.get_info', lambda *args, **kwargs: SOME_PASSPORT_INFO.copy())
    def test_ok(self):
        Phone(
            pk=SOME_PHONE_ID,
            device_id=SOME_DEVICE_ID
        ).save()
        self.data.update({
            'package_name': 'disk',
            'key': 'autoupload',
            'value': 'on'
        })
        response = self.post()
        self.assertEqual(response.status_code, status.HTTP_200_OK)

    @mock.patch('yaphone.advisor.common.passport.get_info', lambda *args, **kwargs: SOME_PASSPORT_INFO.copy())
    def test_disk_autoupload_on(self):
        Phone(pk=SOME_PHONE_ID).save()
        self.data.update({
            'package_name': 'disk',
            'key': 'autoupload',
            'value': 'on'
        })
        self.post()
        phone = Phone.objects.get(pk=SOME_PHONE_ID)
        self.assertTrue(phone.disk_autoupload)

    @mock.patch('yaphone.advisor.common.passport.get_info', lambda *args, **kwargs: SOME_PASSPORT_INFO.copy())
    def test_disk_autoupload_turn_off(self):
        Phone(pk=SOME_PHONE_ID, disk_autoupload=True).save()
        self.data.update({
            'package_name': 'disk',
            'key': 'autoupload',
            'value': 'off'
        })
        self.post()
        phone = Phone.objects.get(pk=SOME_PHONE_ID)
        self.assertFalse(phone.disk_autoupload)

    def test_get_gift_set(self):
        gift_set = GiftSet(passport_uid=SOME_PASSPORT_UID).save()

        passport_uid = SOME_PASSPORT_UID

        self.assertEqual(
            GiftUsedView.get_gift_set(passport_uid),
            gift_set
        )

    def test_get_gift_set_for_taxi(self):
        taxi_promocode = 'some_promo_cod'

        key = 'promo'
        value = taxi_promocode

        gift_set = GiftSet(passport_uid=SOME_PASSPORT_UID).save()
        Phone(
            pk=SOME_PHONE_ID,
            taxi_promocode=taxi_promocode,
            gift_set=gift_set
        ).save()
        self.assertEqual(
            GiftUsedView.get_gift_set_for_taxi(key, value),
            gift_set
        )

    def test_get_gift_set_for_taxi_with_wrong_key(self):
        taxi_promocode = 'some_promo_code'

        key = 'wrong_key'
        value = None

        gift_set = GiftSet(passport_uid=SOME_PASSPORT_UID).save()
        Phone(
            pk=SOME_PHONE_ID,
            taxi_promocode=taxi_promocode,
            gift_set=gift_set
        ).save()
        self.assertIsNone(
            GiftUsedView.get_gift_set_for_taxi(key, value)
        )

    @mock.patch('yaphone.advisor.common.passport.get_info', lambda *args, **kwargs: SOME_PASSPORT_INFO.copy())
    def test_gift_activated(self):
        passport_uid = SOME_PASSPORT_UID
        self.data.update({
            'package_name': 'ru.yandex.money',
            'key': 'contactless_card_state',
            'value': 'ready',
        })

        gift_set = GiftSet(
            passport_uid=SOME_PASSPORT_UID,
            gifts=[GivenGift(gift=MONEY_ID)]
        ).save()
        Phone(pk=SOME_PHONE_ID, gift_set=gift_set).save()

        self.post()
        gift_set = GiftSet.objects.get(passport_uid=passport_uid)
        given_gift = gift_set.gifts[0]

        self.assertTrue(
            given_gift.activated
        )

    @mock.patch('yaphone.advisor.setup_wizard.views.gifts.GiftUsedView.get_passport_info', lambda *args, **kwargs: {})
    def test_gift_used_for_plus(self):
        gift_set = GiftSet(
            passport_uid=SOME_PASSPORT_UID,
            gifts=[GivenGift(gift=PLUS_ID)]
        ).save()
        Phone(pk=SOME_PHONE_ID, gift_set=gift_set).save()

        self.data = {
            'package_name': 'ru.yandex.music',
            'key': 'plus_subscription',
            'value': 'purchased',
            'uid': SOME_PASSPORT_UID,
        }
        response = self.post()
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        # test that Plus was activated
        gift_set = GiftSet.objects.get(passport_uid=SOME_PASSPORT_UID)
        given_gift = gift_set.gifts[0]
        self.assertTrue(
            given_gift.activated
        )


class GiftSetTest(GiftsMixin, TestCase):

    def test_get_given_gift_by_id_ok(self):
        gift_set = GiftSet(
            passport_uid=SOME_PASSPORT_UID,
            gifts=[
                GivenGift(gift=TAXI_ID),
                GivenGift(gift=MONEY_ID),
                GivenGift(gift=PLUS_ID),
            ]
        )
        self.assertEqual(gift_set.get_given_gift_by_id(MONEY_ID), gift_set.gifts[1])

    def test_get_given_gift_by_id_raise_error_if_not_found(self):
        gift_set = GiftSet(
            passport_uid=SOME_PASSPORT_UID,
            gifts=[
                GivenGift(gift=TAXI_ID),
                GivenGift(gift=MONEY_ID),
            ]
        )
        with self.assertRaises(GiftSetError):
            gift_set.get_given_gift_by_id(PLUS_ID)

    def test_get_given_gift_by_id_raise_error_if_gifts_list_is_empty(self):
        gift_set = GiftSet(
            passport_uid=SOME_PASSPORT_UID,
            gifts=[]
        )
        with self.assertRaises(GiftSetError):
            gift_set.get_given_gift_by_id(PLUS_ID)

    def test_activate_gift_ok(self):
        gift_set = GiftSet(
            passport_uid=SOME_PASSPORT_UID,
            gifts=[
                GivenGift(gift=TAXI_ID),
                GivenGift(gift=MONEY_ID),
                GivenGift(gift=PLUS_ID),
            ]
        )

        gift_set.activate_gift(PLUS_ID)
        given_gift = gift_set.get_given_gift_by_id(PLUS_ID)
        self.assertAlmostEqual(given_gift.activated_at, datetime.utcnow(), delta=timedelta(seconds=1))
        self.assertTrue(given_gift.activated)

    def test_multiple_activate_gift_not_affect_activation_time(self):
        gift_set = GiftSet(
            passport_uid=SOME_PASSPORT_UID,
            gifts=[
                GivenGift(gift=PLUS_ID),
            ]
        )

        # first activtion
        gift_set.activate_gift(PLUS_ID)
        given_gift = gift_set.get_given_gift_by_id(PLUS_ID)
        first_activation_time = given_gift.activated_at

        # gift is already activated
        # second activation should not affect activation time
        gift_set.activate_gift(PLUS_ID)
        given_gift = gift_set.get_given_gift_by_id(PLUS_ID)
        second_activation_time = given_gift.activated_at

        self.assertEqual(first_activation_time, second_activation_time)

    def test_add_promocodes(self):
        gift_set = GiftSet(
            passport_uid=SOME_PASSPORT_UID,
            gifts=[
                GivenGift(gift=TAXI_ID),
                GivenGift(gift=MONEY_ID),
                GivenGift(gift=PLUS_ID),
            ]
        )

        taxi_promocode = 'taxi_promocode'
        plus_promocode = 'plus_promocode'
        promocodes = {
            'taxi': taxi_promocode,
            'plus': plus_promocode
        }

        gift_set.add_promocodes(promocodes)
        self.assertEqual(
            plus_promocode,
            gift_set.get_given_gift_by_id(PLUS_ID).promocode
        )
        self.assertEqual(
            taxi_promocode,
            gift_set.get_given_gift_by_id(TAXI_ID).promocode
        )
        self.assertIsNone(gift_set.get_given_gift_by_id(MONEY_ID).promocode)
