import logging
import mongoengine as me
from datetime import datetime
from django.conf import settings
from django.utils import timezone as dtz
from rest_framework.parsers import JSONParser
from rest_framework.renderers import JSONRenderer
from rest_framework.response import Response
from uuid import UUID
from ylog.context import log_context

from yaphone.advisor.advisor.views.base import BaseAPIView, StatelessView, get_user_ip
from yaphone.advisor.common.exceptions import BadRequestAPIError
from yaphone.advisor.common.localization_helpers import translate
from yaphone.advisor.common.passport import user_info_by_login
from yaphone.advisor.common.rest import IgnoreClientContentNegotiation
from yaphone.advisor.common.tools import make_mds_url, to_timestamp
from yaphone.advisor.setup_wizard.models import (Gifts, Phone, GiftSet, GreetingMail, GIFTS_ID_TITLE, TITLE_ID_TEMPLATE,
                                                 DESCRIPTION_ID_TEMPLATE,
                                                 DISK_ID, MONEY_ID, TAXI_ID, PLUS_ID, DEFAULT_DEVICE_NAME)
from yaphone.advisor.setup_wizard.serializers import (BasePhoneQueryValidator, GiftUsedQueryValidator,
                                                      GiftsPreviewQueryValidator, GiftsTransferQueryValidator)

logger = logging.getLogger(__name__)

RUSSIA_REGION_ID = 225
PROMOCODE_LOCK_TTL = 1500  # millisecond
MUSIC_PACKAGE_NAME = 'ru.yandex.music'

GIFTS_TURN_OFF_DEADLINE = dtz.datetime(2020, 1, 1)
TEST_GIFTS_TURN_OFF_PHONE_IDS = ('a00052665b61fcd76ef45690e8dc2844c237644504d86b61c5f74f59e73c0c9f',)


def should_turn_off_gifts(phone):
    return dtz.datetime.now() >= GIFTS_TURN_OFF_DEADLINE or phone.id in TEST_GIFTS_TURN_OFF_PHONE_IDS


def raise_error_and_log_with_context(msg, **kwargs):
    with log_context(**kwargs):
        logger.error(msg)
    raise BadRequestAPIError(msg)


def log_with_context(msg, *args, **kwargs):
    context = {}
    for key in ('phone_id', 'device_id', 'passport_uid'):
        if key in kwargs:
            value = kwargs.pop(key)
            if isinstance(value, UUID):
                value = value.hex
            context[key] = value

    with log_context(**context):
        context_str = u' Context: ' + u'; '.join(map(u'{0[0]}={0[1]}'.format, context.iteritems()))
        logger.info(msg + context_str, *args, **kwargs)


class GiftsError(BadRequestAPIError):
    def __init__(self, title=None, message=None):
        self.detail = {'title': title, 'message': message}


class WrongAccountTypeResponse(Response):
    def __init__(self, title=None, message=None):
        super(WrongAccountTypeResponse, self).__init__(
            data={'title': title, 'message': message},
            status=419,
        )
        self.reason_phrase = "WRONG ACCOUNT TYPE"


class GiftsPreviewView(StatelessView):
    validator_class = GiftsPreviewQueryValidator
    renderer_classes = (JSONRenderer,)

    def get_uuid(self, request):
        """ UUID is not required here """
        try:
            return super(GiftsPreviewView, self).get_uuid(request)
        except BadRequestAPIError:
            return None

    def post(self, request, *args, **kwargs):
        data = self.get_validated_data(request.data)
        return self.prepare_response(data['phone_id'])

    def get(self, request, *args, **kwargs):
        data = self.get_validated_data(request.query_params)
        return self.prepare_response(data['phone_id'])

    def prepare_response(self, phone_id):
        gifts = []
        log_with_context("Turn off all gifts except DISK.", phone_id=phone_id)
        preview_gifts = Gifts.objects.filter(id=DISK_ID)
        log_with_context("Generating response with gifts: %s", preview_gifts, phone_id=phone_id)
        for gift in preview_gifts:
            translate_id = gift.pk
            gifts.append({
                'id': gift.pk,
                'image_url': make_mds_url(gift.suw_preview_image, host=self.host),
                'title': translate(self.client, TITLE_ID_TEMPLATE.format(id=translate_id)),
                'description': translate(self.client, DESCRIPTION_ID_TEMPLATE.format(id=translate_id)),
                'updated_at': gift.updated_at,
            })

        result = {
            'title': translate(self.client, GIFTS_ID_TITLE),
            'gifts': gifts,
        }
        return Response(result)


class ActivationView(StatelessView):
    validator_class = BasePhoneQueryValidator

    def post(self, request, *args, **kwargs):
        data = self.get_validated_data(request.data)
        phone_id = data['phone_id']
        device_id = data['device_id']

        phone = get_phone(phone_id)

        phone.device_id = device_id
        utcnow = datetime.utcnow()
        first_activation = not phone.activated
        if first_activation:
            log_with_context('First phone activation', phone_id=phone_id, device_id=device_id)
            phone.activated = True
            phone.first_activation_datetime = utcnow
        else:
            log_with_context('Non-first phone activation', phone_id=phone_id, device_id=device_id)
        phone.last_activation_datetime = utcnow

        phone.save()
        return Response({
            "timestamp": to_timestamp(utcnow)
        })


class GiveGiftsView(StatelessView):
    validator_class = GiftsPreviewQueryValidator

    def post(self, request, *args, **kwargs):
        data = self.get_validated_data(request.data)
        passport_info = self.get_passport_info(request)

        phone_id = data['phone_id']
        device_id = data['device_id']

        device_name = '{} {}'.format(self.user_agent.device_manufacturer, self.user_agent.device_model)
        return give_gifts(phone_id=phone_id, device_id=device_id,
                          passport_info=passport_info, country_id=self.country_id,
                          device_name=device_name, client=self.client)


def give_gifts(phone_id, device_id, passport_info, country_id,
               device_name=DEFAULT_DEVICE_NAME, client=None, log_error_on_existing_greeting_mail=True):
    has_plus = passport_info['has_plus']
    passport_uid = GiveGiftsView.get_uid_from_passport_info(passport_info)
    log_with_context("Stopped giving gifts. No gifts ever.", phone_id=phone_id,
                     device_id=device_id, passport_uid=passport_uid)
    return Response({
        'give_gifts': False,
        'has_plus': has_plus,
    })


class GiftUsedView(StatelessView):
    validator_class = GiftUsedQueryValidator
    parser_classes = (JSONParser,)
    renderer_classes = (JSONRenderer,)
    # overriding negotiation class to always use JSONParser and JSONRenderer
    content_negotiation_class = IgnoreClientContentNegotiation

    def initial(self, request, *args, **kwargs):
        # Music sends gift_used from their own backend therefore it does not have headers X-YaClid1,
        # X-YaUuid and etc. Therefore skipping logic that uses these headers.
        if request.data.get('package_name') == MUSIC_PACKAGE_NAME:
            super(BaseAPIView, self).initial(request, *args, **kwargs)
            self.ip = get_user_ip(request)
        else:
            super(GiftUsedView, self).initial(request, *args, **kwargs)

    def post(self, request, *args, **kwargs):
        passport_info = self.get_passport_info(request)
        try:
            passport_uid = self.get_uid_from_passport_info(passport_info)
        except BadRequestAPIError:
            passport_uid = None  # taxi and plus does not send OAuth token
        data = self.get_validated_data(request.data)

        package_name = data['package_name']
        key = data['key'].lower()
        value = data['value']
        phone_id = data.get('phone_id')

        log_with_context('Gift used: package_name=%s, key=%s, value=%s', package_name, key, value,
                         phone_id=phone_id, passport_uid=passport_uid)
        #  Music sends passport_uid in directly POST data, not via OAuth token in Authorization header
        if package_name == MUSIC_PACKAGE_NAME:
            passport_uid = data.get('uid')
            if not passport_uid:
                raise BadRequestAPIError('Invalid request: uid parameter should not be empty')

        if package_name == DISK_ID:  # Disk doesn't send real package_name, only 'disk'
            # Disk is not a real gift. It is just a parameter in Phone model.
            if key == 'autoupload':
                self.update_disk_autoupload(phone_id, value)
        else:
            gift = get_gift_by_package_name(package_name)
            if gift is None:
                raise_error_and_log_with_context(
                    "Can't find Gift by package_name: {}".format(package_name),
                    phone_id=phone_id, passport_uid=passport_uid
                )

            if self.gift_is_activated(gift.pk, key, value):
                if gift.pk == 'taxi':
                    gift_set = self.get_gift_set_for_taxi(key, value)
                else:
                    gift_set = self.get_gift_set(passport_uid)

                phone = Phone.objects.get(gift_set=gift_set)

                # activating only those gifts that are allowed on device
                if gift.pk in phone.allowed_gifts:
                    try:
                        gift_set.activate_gift(gift.pk)
                        gift_set.save()
                    except ValueError as e:
                        logger.error(e)
                else:
                    log_with_context("Can't activate Gift: {}. It is not allowed on this device.".format(gift.pk),
                                     phone_id=phone.pk)

        return Response()

    @staticmethod
    def get_gift_set(passport_uid):
        try:
            return GiftSet.objects.get(passport_uid=passport_uid)
        except me.DoesNotExist:
            raise_error_and_log_with_context("Can't find GiftSet by uid", passport_uid=passport_uid)

    @staticmethod
    def get_gift_set_for_taxi(key, value):
        # taxi does not send OAuth token, so get GiftSet by Phone through promocode
        if key == 'promo':
            try:
                phone = Phone.objects.get(taxi_promocode=value)
                if phone.gift_set:
                    return phone.gift_set.fetch()
            except me.DoesNotExist:
                raise_error_and_log_with_context("Can't find GiftSet by taxi promocode", promocode=value)

    @staticmethod
    def gift_is_activated(gift_id, key, value):
        if gift_id == MONEY_ID:
            return key == 'contactless_card_state' and value == 'ready'
        elif gift_id == TAXI_ID:
            return key == 'promo'
        elif gift_id == PLUS_ID:
            return key == 'plus_subscription' and value == 'purchased'
        return False

    @staticmethod
    def update_disk_autoupload(phone_id, value):
        if not value or value == 'off':
            autoupload = False
        elif value == 'on':
            autoupload = True
        else:
            raise_error_and_log_with_context(
                'Unknown value for disk autoupload: {}'.format(value),
                phone_id=phone_id
            )
        phone = get_phone(phone_id)
        phone.disk_autoupload = autoupload
        phone.save()


def get_phone(phone_id):
    try:
        return Phone.objects.get(pk=phone_id)
    except me.DoesNotExist:
        raise_error_and_log_with_context("Unknown phone_id", phone_id=phone_id)


def get_gift_by_package_name(package_name):
    try:
        gift_package_name = '{prefix}{package_name}'.format(
            prefix=settings.GIFT_PACKAGE_NAME_PREFIX,
            package_name=package_name
        )
        return Gifts.objects.get(package_name=gift_package_name)
    except me.DoesNotExist:
        return


class GiftsTransferView(StatelessView):
    validator_class = GiftsTransferQueryValidator

    def post(self, request, *args, **kwargs):
        data = self.get_validated_data(request.data)
        phone_id = data['phone_id']
        login = data['login']

        phone = get_phone(phone_id)
        device_id = phone.device_id
        if not phone.gift_set:
            msg = 'No gifts are given on this device: {}'.format(phone_id)
            log_with_context(msg, phone_id=phone_id, device_id=device_id)
            raise BadRequestAPIError(detail=msg)

        passport_info = user_info_by_login(login)
        if not passport_info:
            msg = 'No user found by login: {}'.format(login)
            log_with_context(msg, phone_id=phone_id, device_id=device_id)
            raise BadRequestAPIError(detail=msg)

        new_passport_uid = passport_info['passport_uid']

        # saving gift_set to variable to restore it if give_gifts fails
        previous_passport_uid = phone.gift_set.passport_uid
        previous_gift_set = GiftSet.objects(passport_uid=previous_passport_uid)

        # should delete emails for new_user if give_gifts successful
        new_user_greeting_mails_ids = [item.passport_uid for item in
                                       GreetingMail.objects(passport_uid=new_passport_uid).only('passport_uid')]

        log_with_context('Trying to transfer gifts to new user: {}'.format(new_passport_uid),
                         phone_id=phone_id, device_id=device_id, passport_uid=previous_passport_uid)
        phone.gift_set = None
        phone.save()
        try:
            response = give_gifts(phone_id=phone_id, device_id=device_id, passport_info=passport_info,
                                  country_id=RUSSIA_REGION_ID, log_error_on_existing_greeting_mail=False)

            if not response.data.get('give_gifts', False):
                raise ValueError('Give_gifts returned False')

        except Exception as e:
            msg = 'Failed to transfer gifts due to error in give_gifts'
            log_with_context(msg)
            logger.exception(e)

            phone.gift_set = previous_gift_set
            phone.save()
            return Response({
                'status': 'error',
                'message': msg,
            })

        previous_gift_set.delete()
        GreetingMail.objects(passport_uid__in=new_user_greeting_mails_ids).delete()

        log_with_context('Successfully transfered gifts to new user: {}'.format(new_passport_uid),
                         phone_id=phone_id, device_id=device_id, passport_uid=previous_passport_uid)

        return Response({
            'status': 'ok',
        })
