# -*- coding: utf-8 -*-

import ujson as json

import logging
import requests

from yaphone.advisor.common.tools import make_requests_session

logger = logging.getLogger(__name__)


class WeatherError(Exception):
    pass


class WeatherResultBase(object):
    def __init__(self, data):
        self.data = data

    def get_tzinfo(self):
        raise NotImplementedError

    def get_city_name(self):
        raise NotImplementedError

    def get_current_weather(self):
        raise NotImplementedError

    def get_forecast_by_hours(self):
        raise NotImplementedError

    def get_forecast_by_parts(self, count):
        raise NotImplementedError

    def get_forecast_by_days(self, count):
        raise NotImplementedError

    def get_latitude(self):
        raise NotImplementedError

    def get_longitude(self):
        raise NotImplementedError

    def get_region_id(self):
        raise NotImplementedError


# noinspection PyAbstractClass
class WeatherResultJsonBase(WeatherResultBase):
    def __init__(self, data):
        super(WeatherResultJsonBase, self).__init__(data)


class WeatherClientBase(object):
    result_class = WeatherResultBase
    http = make_requests_session(pool_maxsize=100)

    # noinspection PyUnusedLocal
    def __init__(self, url, headers=None, *args, **kwargs):
        self.url = url
        self.headers = headers

    def _get_params(self, **kwargs):
        return kwargs

    def _get_response(self, **kwargs):
        url = kwargs.pop('url', None) or self.url
        try:
            response = self.http.get(url, params=self._get_params(**kwargs),
                                     allow_redirects=False, headers=self.headers)
            if response.status_code == 404:
                return None
            response.raise_for_status()
            response_content = response.content
            return json.loads(response_content)
        except (requests.exceptions.RequestException, requests.exceptions.BaseHTTPError) as e:
            raise WeatherError('Failed request to "%s": %s' % (url, e))

    def get_forecast_by_ll(self, lat, lon, locale):
        data = self._get_response(lat=lat, lon=lon, locale=locale)
        return self.result_class(data) if data else None
