# -*- coding: utf-8 -*-

import logging
from datetime import datetime
from django.conf import settings
from rest_framework.exceptions import NotFound, ValidationError
from rest_framework.response import Response
from rest_framework.serializers import FloatField, Serializer, CharField

from yaphone.advisor.advisor.views.base import MobileApiView, StatelessView
from yaphone.advisor.common.android_tz import AndroidTimezone
from yaphone.advisor.common.localization_helpers import get_impersonal_config_value
from yaphone.advisor.launcher.validators import TimezoneField, LbsInfoValidator, LocaleTimezoneValidator
from yaphone.advisor.weather.base import WeatherError
from yaphone.advisor.weather.yandex_pogoda import YandexPogodaClient
from yaphone.utils import geo

logger = logging.getLogger(__name__)

WEATHER_CONDITION_MAP_KEY = 'weather_condition_map'

KNOWN_CONDITIONS = frozenset([
    "clear",
    "partly-cloudy",
    "cloudy",
    "overcast",
    "partly-cloudy-and-light-rain",
    "cloudy-and-light-rain",
    "overcast-and-light-rain",
    "partly-cloudy-and-rain",
    "cloudy-and-rain",
    "overcast-and-rain",
    "overcast-thunderstorms-with-rain",
    "overcast-and-wet-snow",
    "partly-cloudy-and-light-snow",
    "cloudy-and-light-snow",
    "overcast-and-light-snow",
    "partly-cloudy-and-snow",
    "cloudy-and-snow",
    "overcast-and-snow",
])

yandex_weather_provider = YandexPogodaClient(settings.WEATHER['api_url'],
                                             headers={'X-Yandex-API-Key': settings.WEATHER['api_key']})


class WeatherInvalidConditionError(WeatherError):
    pass


def translate_condition(condition):
    if condition in KNOWN_CONDITIONS:
        return condition

    conditions_map = get_impersonal_config_value(WEATHER_CONDITION_MAP_KEY)
    if isinstance(conditions_map, dict):
        if condition in conditions_map:
            logger.warning('in the localization db in the key %r has been found substitution for new condition %r',
                           WEATHER_CONDITION_MAP_KEY, condition)
            return conditions_map[condition]
    else:
        logger.error('in the localization db for key %r must be defined json dict object',
                     WEATHER_CONDITION_MAP_KEY)

    raise WeatherInvalidConditionError(
        'unknown weather condition: %r. please, use localization key %r to define substitution' % (
            condition, WEATHER_CONDITION_MAP_KEY))


class FloatOrNoneField(FloatField):
    def to_internal_value(self, data):
        try:
            return super(FloatOrNoneField, self).to_internal_value(data)
        except ValidationError:
            return None


# noinspection PyAbstractClass
class WeatherForecastValidator(Serializer):
    lat = FloatOrNoneField(required=False)
    lon = FloatOrNoneField(required=False)
    locale = CharField(required=True)
    timezone = TimezoneField(required=False)


# noinspection PyAbstractClass
class WeatherForecastValidatorV2(Serializer):
    lat = FloatField(required=True, min_value=-90.0, max_value=90.0)
    lon = FloatField(required=True, min_value=-180.0, max_value=180.0)
    timezone = TimezoneField(required=False)


def translate_values(weather, time_offset):
    """ current version translates just a condition """
    weather['condition'] = translate_condition(weather['condition'])
    weather['dt'] -= time_offset
    return weather


def get_tz_offset(tz_src, tz_dst):
    dt = datetime.utcnow()
    tz_src = AndroidTimezone(tz_src)
    tz_dst = AndroidTimezone(tz_dst)
    return int((tz_dst.utcoffset(dt, True) - tz_src.utcoffset(dt, True)).total_seconds())


def get_weather(lat, lon, locale, tz_dst=None, accuracy=0):
    # noinspection PyBroadException
    try:
        forecast = yandex_weather_provider.get_forecast_by_ll(lat, lon, locale=locale)
        if not forecast:
            logger.warning('No weather data: lat=%s, lon=%d, locale=%s, tz_dst=%s', lat, lon, locale, tz_dst)
            raise NotFound(detail='No data')
        hours_forecast = forecast.get_forecast_by_hours()
        parts_forecast = forecast.get_forecast_by_parts(3)
        days_forecast = forecast.get_forecast_by_days(7)
        detected_tz_name = forecast.get_tzinfo().get('name', 0)
        time_offset = get_tz_offset(detected_tz_name, tz_dst or detected_tz_name)
        logger.info('detected_tz_name=%s, time_offset=%d', detected_tz_name, time_offset)
        result = {
            'time_zone': {
                'name_detected': detected_tz_name,
                'name_passed': tz_dst,
                'offset': time_offset,
            },
            'city': {
                'name': forecast.get_city_name(),
                'region_id': forecast.get_region_id()
            },
            'current': translate_values(forecast.get_current_weather(), time_offset),
            'forecast': {
                'hours': [translate_values(w, time_offset) for w in hours_forecast[:4]],
                'parts': [translate_values(w, time_offset) for w in parts_forecast],
                'days': [translate_values(w, time_offset) for w in days_forecast]
            },
            'position': {
                'lat': forecast.get_latitude(),
                'lon': forecast.get_longitude(),
                'accuracy': accuracy
            }
        }
        return result
    except WeatherInvalidConditionError:
        raise  # will cause http 500 and will be logged in middleware
    except WeatherError as e:
        logger.warning('Request to weather API failed', extra={'error_message': e.message})
        raise NotFound(detail='No data')


def get_location_from_lbs_info(lbs_info, **kwargs):
    if lbs_info['location']:
        return dict(lat=lbs_info['location']['latitude'],
                    lon=lbs_info['location']['longitude'])

    lbs_location = geo.lbs_locator.locate(
        gsm_cells=lbs_info['cells'],
        wifi_networks=lbs_info['wifi_networks'],
        **kwargs
    )
    if lbs_location and lbs_location['type'] != 'ip':
        return dict(lat=lbs_location['latitude'],
                    lon=lbs_location['longitude'],
                    accuracy=lbs_location['precision'])


class WeatherForecastView(MobileApiView):
    throttle_scope = 'weather'
    validator_class = WeatherForecastValidator

    def get(self, request):
        validated_data = self.get_validated_data(request.query_params)

        lat = validated_data.get('lat')
        lon = validated_data.get('lon')
        locale = validated_data['locale']
        timezone = validated_data.get('timezone')

        if lat is None or lon is None:
            if self.client.profile.lbs_info.location:
                lat = self.client.profile.lbs_info.location.latitude
                lon = self.client.profile.lbs_info.location.longitude
            else:
                raise NotFound(detail='Could not determine user location')
        return Response(get_weather(lat, lon, locale, timezone))


class WeatherForecastViewV2(StatelessView):
    throttle_scope = 'weather'
    validator_class = WeatherForecastValidatorV2

    def get(self, request):
        validated_data = self.get_validated_data(request.query_params)

        lat = validated_data['lat']
        lon = validated_data['lon']
        locale = str(self.client.locale)
        timezone = validated_data.get('timezone')

        return Response(get_weather(lat, lon, locale, timezone))

    def post(self, request):
        logger.info('Weather request: uuid=%s\trequest=%s', self.uuid.hex, request.body)

        # validate parameters from URL
        query_data = self.get_validated_data(request.query_params, LocaleTimezoneValidator)
        locale = query_data.get('locale', str(self.client.locale))
        timezone = query_data.get('timezone')

        # validate POST parameters
        lbs_info = self.get_validated_data(request.data, LbsInfoValidator)

        location = get_location_from_lbs_info(lbs_info, uuid=self.uuid, ip=self.ip)

        if location:
            return Response(get_weather(locale=locale, tz_dst=timezone, **location))
        else:
            raise NotFound(detail='Could not determine user location')
