# -*- coding: utf-8 -*-

import logging
from datetime import datetime, timedelta
from operator import itemgetter

from yaphone.advisor.common.tools import to_timestamp
from yaphone.advisor.weather.base import WeatherClientBase, WeatherResultJsonBase

logger = logging.getLogger(__name__)


class YandexPogodaResult(WeatherResultJsonBase):
    parts_hours_map = {
        'night': 0,
        'morning': 6,
        'day': 12,
        'evening': 18,
    }

    def get_tzinfo(self):
        return self.data['info'].get('tzinfo', {})

    def get_latitude(self):
        return self.data['info'].get('lat')

    def get_longitude(self):
        return self.data['info'].get('lon')

    def get_city_name(self):
        locality = self.data['geo_object'].get('locality') or {}
        return locality.get('name')

    def get_region_id(self):
        locality = self.data['geo_object'].get('locality') or {}
        return locality.get('id')

    def get_current_weather(self):
        current = self.data['fact']
        weather = dict(temp=dict(avg=current['temp'], feels_like=current['feels_like']),
                       wind=dict(speed=current['wind_speed']),
                       condition=current['condition'],
                       dt=current['obs_time'],
                       humidity=current.get('humidity'),
                       is_night=current['icon'].endswith('_n'),
                       )
        return weather

    def get_forecast_by_parts(self, count):
        result = []
        now = datetime.utcnow()
        for day in self.data['forecasts']:
            if 'parts' in day:
                day_date = datetime.utcfromtimestamp(day['date_ts'])
                for name, part in day['parts'].items():
                    if name not in self.parts_hours_map:
                        continue
                    dt = day_date + timedelta(hours=self.parts_hours_map[name])
                    if dt >= now:
                        weather = dict(
                            temp=dict(avg=part['temp_avg'], feels_like=part['feels_like']),
                            wind=dict(speed=part.get('wind_speed')),
                            condition=part['condition'],
                            dt=to_timestamp(dt),
                            humidity=part.get('humidity'),
                            is_night=part['icon'].endswith('_n'),
                        )
                        result.append(weather)
        result.sort(key=itemgetter('dt'))
        return result[:count]

    def get_forecast_by_days(self, count):
        result = []
        for day in self.data['forecasts']:
            if 'parts' in day and 'day_short' in day['parts']:
                day_short = day['parts']['day_short']
                weather = dict(
                    temp=dict(avg=day_short['temp'], feels_like=day_short['feels_like']),
                    wind=dict(speed=day_short.get('wind_speed')),
                    condition=day_short['condition'],
                    dt=day['date_ts'],
                    humidity=day_short['humidity'],
                    is_night=day_short['icon'].endswith('_n'),
                )
                result.append(weather)
        result.sort(key=itemgetter('dt'))
        return result

    def get_forecast_by_hours(self):
        result = []
        for day in self.data['forecasts']:
            if 'hours' in day:
                date = datetime.strptime(day['date'], '%Y-%m-%d')
                for item in day['hours']:
                    weather = dict(
                        temp=dict(avg=item['temp'], feels_like=item['feels_like']),
                        wind=dict(speed=item.get('wind_speed')),
                        condition=item['condition'],
                        dt=to_timestamp(date.replace(hour=int(item['hour']))),
                        humidity=item.get('humidity'),
                        is_night=item['icon'].endswith('_n'),
                    )
                    result.append(weather)
        now = to_timestamp(datetime.utcnow())
        result = filter(lambda v: v['dt'] >= now, result)
        result.sort(key=itemgetter('dt'))
        return result


class YandexPogodaClient(WeatherClientBase):
    result_class = YandexPogodaResult
    supported_locales = {'ru_RU', 'uk_UA', 'be_BY', 'en_US', 'tr_TR'}
    default_locale = 'en_US'

    def _get_params(self, **kwargs):
        locale = kwargs.pop('locale', 'en_US')
        if locale not in self.supported_locales:
            locale = self.default_locale
        kwargs['lang'] = locale
        return kwargs
