import math
from copy import copy

__all__ = ['ColorInfo']


def j_round(f):
    return int(math.floor(f + 0.5))


def constrain(amount, low, high):
    return low if amount < low else (high if amount > high else amount)


def color_to_hsl(color):
    rf = color[0] / 255.0
    gf = color[1] / 255.0
    bf = color[2] / 255.0
    max_c = max(rf, gf, bf)
    min_c = min(rf, gf, bf)
    delta_max_min = max_c - min_c
    l = (max_c + min_c) / 2.0
    if max_c == min_c:
        # Monochromatic
        h = s = 0.0
    else:
        if max_c == rf:
            h = ((gf - bf) / delta_max_min) % 6.0
        elif max_c == gf:
            h = ((bf - rf) / delta_max_min) + 2.0
        else:
            h = ((rf - gf) / delta_max_min) + 4.0
        s = delta_max_min / (1.0 - abs(2.0 * l - 1.0))

    h = (h * 60.0) % 360.0
    if h < 0:
        h += 360.0

    h = constrain(h, 0.0, 360.0)
    s = constrain(s, 0.0, 1.0)
    l = constrain(l, 0.0, 1.0)

    return [h, s, l]


def hsl_to_color(hsl):
    h, s, l = hsl
    c = (1.0 - abs(2 * l - 1.0)) * s
    m = l - 0.5 * c
    x = c * (1.0 - abs((h / 60.0 % 2.0) - 1.0))
    hue_segment = int(h / 60)
    r = g = b = 0

    if hue_segment == 0:
        r = j_round(255 * (c + m))
        g = j_round(255 * (x + m))
        b = j_round(255 * m)
    elif hue_segment == 1:
        r = j_round(255 * (x + m))
        g = j_round(255 * (c + m))
        b = j_round(255 * m)
    elif hue_segment == 2:
        r = j_round(255 * m)
        g = j_round(255 * (c + m))
        b = j_round(255 * (x + m))
    elif hue_segment == 3:
        r = j_round(255 * m)
        g = j_round(255 * (x + m))
        b = j_round(255 * (c + m))
    elif hue_segment == 4:
        r = j_round(255 * (x + m))
        g = j_round(255 * m)
        b = j_round(255 * (c + m))
    elif hue_segment in (5, 6):
        r = j_round(255 * (c + m))
        g = j_round(255 * m)
        b = j_round(255 * (x + m))

    r = constrain(r, 0, 255)
    g = constrain(g, 0, 255)
    b = constrain(b, 0, 255)

    return [r, g, b]


class ColorInfo(object):
    def __init__(self, color):
        self.hsl = copy(color.hsl) if isinstance(color, ColorInfo) else color_to_hsl(color)

    def __str__(self):
        return '#%02X%02X%02X' % tuple(self.get_color())

    def __repr__(self):
        return str(self)

    def get_color(self):
        return hsl_to_color(self.hsl)

    def is_dark(self):
        return (self.hsl[2] < 0.4) if self.is_yellow() else (self.hsl[2] < 0.55)

    def get_adapted_l(self):
        return (self.hsl[2] * 1.5) if self.is_yellow() else self.hsl[2]

    def is_yellow(self):
        return 30.0 <= self.hsl[0] < 95.0

    @staticmethod
    def compare_l(c0, c1):
        # noinspection PyCompatibility
        return cmp(c0.get_adapted_l(), c1.get_adapted_l())
