"""
https://github.yandex-team.ru/kovchiy/colorwiz/blob/master/colorwiz.js
https://github.yandex-team.ru/moblauncher/yandex_launcher/blob/next-release/src/com/yandex/launcher/util/RecColors.java
"""

# !/usr/bin/env python
# -*- coding: utf-8 -*-

import math

from color_info import ColorInfo
from uniform_color_detector import UniformColorDetector

COLOR_TRANSPARENT = 1
COLOR_WHITE = 2
COLOR_BLACK = 4
COLOR_WHITE_OR_BLACK = COLOR_WHITE | COLOR_BLACK

COLOR_OPAQUE = 0xFF


def alpha(c):
    return c[3] if len(c) > 3 else COLOR_OPAQUE


def rgba(c):
    return (c[0], c[1], c[2], c[3]) if len(c) > 3 else (c[0], c[1], c[2], COLOR_OPAQUE)


def get_stroke_color(data, width, height):
    if width < 4 or height < 4:
        return

    color = data[width + 1]  # (1, 1)
    if alpha(color) < 128:
        return

    if (not has_rect_color(data, 1, 1, width - 2, 1, color, width) or
            not has_rect_color(data, 1, height - 2, width - 2, 1, color, width) or
            not has_rect_color(data, 1, 1, 1, height - 2, color, width) or
            not has_rect_color(data, width - 2, 1, 1, height - 2, color, width)):
        return

    return ColorInfo(color)


BUTTON_DELTA = 0.2
BACKGROUND_DELTA = 0.15
BUTTON_TEXT_DELTA = 0.6


def calculate(data, width, height):
    # Check if the bitmap has uniform background and use it afterwards
    background_color = get_stroke_color(data, width, height)
    uniform_color_detector = UniformColorDetector()

    # Count pixels w/o transparent, white, black
    wot_counter = wot_wb_counter = 0
    for color in data:
        color_type = get_color_type(color)
        if color_type & COLOR_TRANSPARENT == 0:
            wot_counter += 1
            if color_type & COLOR_WHITE_OR_BLACK == 0:
                wot_wb_counter += 1

            uniform_color_detector.register(color)

    # Find 4 initial pixels
    use_wb = wot_wb_counter < 0.12 * len(data)
    counter = wot_counter if use_wb else wot_wb_counter
    mask = COLOR_TRANSPARENT if use_wb else (COLOR_TRANSPARENT | COLOR_WHITE_OR_BLACK)
    has_uniform_color = uniform_color_detector.has_uniform_color(0.96)

    color_info = []
    dark_count = 0

    side = int(math.sqrt(counter))
    if side > 1:
        y0 = side / 10
        y1 = side / 2 + 1
        x0 = side / 4
        x1 = 3 * side / 4

        point = (
            y0 * side + x0,
            y0 * side + x1,
            y1 * side + x0,
            y1 * side + x1
        )
        len_point = len(point)

        r = p = 0
        for color in data:
            color_type = get_color_type(color)
            if (color_type & mask) != 0:
                continue

            if r >= point[p]:
                ci = ColorInfo(color)
                if ci.is_dark():
                    dark_count += 1

                color_info.append(ci)
                p += 1
                if p >= len_point:
                    break
            r += 1

        color_info = sorted(color_info, cmp=ColorInfo.compare_l)

    ci_len = len(color_info)
    c0 = color_info[0] if ci_len > 0 else ColorInfo((0xff, 0xff, 0xff, 0xff))
    c1 = color_info[1] if ci_len > 1 else c0
    c2 = color_info[2] if ci_len > 2 else c1
    c3 = color_info[3] if ci_len > 3 else c2

    is_dark = background_color.is_dark() if background_color is not None else (dark_count > 1)

    # Manage colors
    if is_dark:  # Dark scheme
        if background_color is not None:
            card_color = ColorInfo(background_color)
        else:
            card_color = ColorInfo(c0 if c0.hsl[2] >= 0.1 else c1)

        if card_color.hsl[2] < 0.1:
            card_color.hsl[2] = 0.1

        if has_uniform_color:  # Modify background to show the shape drawn
            if card_color.hsl[2] >= 0.1 + BACKGROUND_DELTA:
                card_color.hsl[2] -= BACKGROUND_DELTA
            else:
                card_color.hsl[2] += BACKGROUND_DELTA

        text_color = ColorInfo(c3)
        if abs(text_color.hsl[2] - card_color.hsl[2]) < 0.6:
            text_color.hsl[2] = min(text_color.hsl[2] + 0.5, 1.0)

        button_color = ColorInfo(c1)
        if abs(button_color.hsl[2] - card_color.hsl[2]) < BUTTON_DELTA:
            button_color = ColorInfo(c2)
            if abs(button_color.hsl[2] - card_color.hsl[2]) < BUTTON_DELTA:
                button_color.hsl[2] = (button_color.hsl[2] + BUTTON_DELTA) \
                    if button_color.hsl[2] <= 1.0 - BUTTON_DELTA else button_color.hsl[2] - BUTTON_DELTA

    else:  # Use light scheme
        if background_color is not None:
            card_color = ColorInfo(background_color)
        else:
            card_color = ColorInfo(c3)

        if has_uniform_color:  # Modify background to show the shape drawn
            if card_color.hsl[2] >= 0.1 + BACKGROUND_DELTA:
                card_color.hsl[2] -= BACKGROUND_DELTA
            else:
                card_color.hsl[2] += BACKGROUND_DELTA

        text_color = ColorInfo(c0)
        if abs(text_color.hsl[2] - card_color.hsl[2]) < 0.6:
            text_color.hsl[2] = max(text_color.hsl[2] - 0.5, 0.0)

        button_color = ColorInfo(c2)
        if abs(button_color.hsl[2] - card_color.hsl[2]) < BUTTON_DELTA:
            button_color = ColorInfo(c1)
            if abs(button_color.hsl[2] - card_color.hsl[2]) < BUTTON_DELTA:
                button_color.hsl[2] = (button_color.hsl[2] - BUTTON_DELTA) \
                    if button_color.hsl[2] >= BUTTON_DELTA else button_color.hsl[2] + BUTTON_DELTA

    button_text_color = ColorInfo(button_color)
    if button_text_color.is_dark():
        button_text_color.hsl[2] = min(button_color.hsl[2] + BUTTON_TEXT_DELTA, 1.0)
    else:
        button_text_color.hsl[2] = max(button_color.hsl[2] - BUTTON_TEXT_DELTA, 0.0)

    return {
        "card_background": str(card_color),
        "card_text": str(text_color),
        "button_background": str(button_color),
        "button_text": str(button_text_color)
    }


def get_color_type(color):
    if alpha(color) < 128:
        return COLOR_TRANSPARENT

    r = color[0]
    g = color[1]
    b = color[2]
    if r > 250 and g > 250 and b > 250:
        return COLOR_WHITE

    if r < 5 and g < 5 and b < 5:
        return COLOR_BLACK

    return 0


# noinspection PyCompatibility
def has_rect_color(pixels, x0, y0, w, h, color, img_w):
    length = len(color)
    for y in xrange(y0, y0 + h):
        i0 = y * img_w + x0
        for i in xrange(i0, i0 + w):
            c = pixels[i]
            for j in xrange(length):
                if abs(c[j] - color[j]) > 3:
                    return False
    return True
