import logging
import urllib
from urlparse import urlparse

import re
from nile.api.v1 import Record

logger = logging.getLogger(__name__)

POSTBACK_PATH_REGEX = re.compile('/api/v1/postback/(?P<adnetwork>[\w-]+)/?')

IMPRESSION_ID_PARAM_ID = '736'  # id of impression_id param to extract from queryargs of bs-hit-log


class PostbackParser(object):
    _adnetwork_to_parameter_name = {
        'mobusi': 'a',
        'applift': 'transaction_id',
        'artofclick': 'aff_sub',
        'mobave': 'aff_sub',
        'unilead': 'aff_sub',
        'yeahmobi': 'aff_sub',
        'minimob': 'clickid',
        'startapp': 'clickId',
        'appnext': 'impressionid',
        'supersonic': 'ImpressionId',
        'ironsource': 'impression_id',
        'taptica': 'impression_id',
        'glispa': 'sid',
        'personaly': 'subid1',
        'youappi': 'sub_id',
        'pubnative': 'transaction_id',
        'cheetah': 'impression_id',
        'mobupps': 'transaction_id',
        'wadogo': 'impression_id',
        'rocket10': 'impression_id',
        'mundomedia': 'impression_id',
        'gameberry': 'impression_id',
    }

    _direct_adnetwork_prefix = 'direct-'

    # All adnetworks that provide not USD offers must have currency and should be listed here
    _must_have_currency_set = frozenset([
        'mobupps', 'unilead', 'wadogo', 'rocket10', 'mobave',
    ])

    def is_direct(self, adnetwork):
        return adnetwork.startswith(self._direct_adnetwork_prefix)

    def must_have_currency(self, adnetwork):
        return adnetwork in self._must_have_currency_set or self.is_direct(adnetwork)

    def impression_id_from_params(self, req_params):
        def postprocessing(result):
            if result is not None:
                # in appnext postbacks 'impressionid' parameter looks like:
                # 'impressionID=<real impression id>'

                if adnetwork == 'appnext':
                    return result[len('impressionID='):]
                elif adnetwork == 'mundomedia':
                    return urllib.unquote(result)
            return result

        adnetwork = req_params['adnetwork']

        if self.is_direct(adnetwork):
            # Direct adnetworks are configured dynamically and have the same impression_id_param_name.
            # Thus, they are not listed in the mapping.
            impression_id_param_name = 'impression_id'
        else:
            impression_id_param_name = self._adnetwork_to_parameter_name.get(adnetwork)

        if adnetwork == 'cheetah':
            # Cheetah limits its params length to 64 bytes max. Collecting impression_id chunks in one string
            # for more info see https://st.yandex-team.ru/ADVISOR-1139
            impression_id_param_names = [impression_id_param_name, 'sub2', 'sub3', 'sub4']
            chunks = [req_params.get(param, '') for param in impression_id_param_names]
            impression_id = ''.join(chunks)
        else:
            impression_id = req_params.get(impression_id_param_name, None)
        return postprocessing(impression_id)

    def parse_postback(self, request):
        req_params = self.parse_postback_request_params(request)
        if req_params is None:
            logger.error('%r request cannot be parsed as postback', request)
            return

        adnetwork = req_params['adnetwork']

        # ignore postbacks from unknown networks
        impression_id = self.impression_id_from_params(req_params)
        if not impression_id:
            logger.error('Could not extract Impression ID from params %r', req_params)
            return

        payout = req_params.get('payout') or req_params.get('revenue')
        try:
            payout = float(str(payout))
        except ValueError:
            logger.warning('Cannot parse payout "%r" from request %s', payout, request)
            payout = None

        currency = req_params.get('currency')
        if currency:
            currency = currency.upper()  # TODO: check currency is valid
        elif self.must_have_currency(adnetwork):
            logger.error('Adnetwork %s must have non-empty currency', adnetwork.capitalize())
            currency = None
        else:
            currency = 'USD'

        return {
            'adnetwork': adnetwork,
            'impression_id': impression_id,
            'payout': payout,
            'currency': currency,
            'click_ip': req_params.get('click_ip'),
        }

    @staticmethod
    def parse_postback_request_params(request):
        req_parsed = urlparse(request)

        query = req_parsed.query
        if not query:
            return
        query = urllib.unquote_plus(query)
        query = query.replace('\\', '')  # startapp

        match = POSTBACK_PATH_REGEX.match(req_parsed.path)
        if not match:
            return
        parsed_params = match.groupdict()

        # parse parameters manually since urlparse.parse_qs split it in a middle of impression id
        for param in query.split('&'):
            # supersonic pass payout in a such strange way
            if param.startswith('$'):
                parsed_params['payout'] = float(param[1:]) / 100.0
                continue

            try:
                key, value = param.split('=', 1)
            except ValueError:
                return

            parsed_params[key] = value
        return parsed_params

    def __call__(self, rows):
        for row in rows:
            result = self.parse_postback(row['request'])
            if result:
                yield Record(row, **result)


def parse_impression_id(impression_id):
    try:
        impression_id_parts = impression_id.split(':')
    except AttributeError:  # in nile task impression_id can be None
        impression_id_parts = []

    return {
        'experiment': impression_id_parts[1] if len(impression_id_parts) >= 2 else '',
        'clid': impression_id_parts[7] if len(impression_id_parts) >= 8 else ''
    }


def postback_is_testing(impression_id):
    impression_id_parts = impression_id.split(':')
    try:
        client_version = impression_id_parts[8].lower()
    except IndexError:
        client_version = ''
    return any([
        'qa' in client_version,
        'dev' in client_version,
    ])


def get_impression_id_from_hitlog(query):
    from urlparse import parse_qs

    if query is None:
        return None

    params = parse_qs(query)
    try:
        return params[IMPRESSION_ID_PARAM_ID][0]
    except KeyError:
        return None
