# -*- coding: utf-8 -*-
import logging
import traceback
from collections import defaultdict
from decimal import Decimal
from operator import itemgetter

import requests
from yacurrency import get_currency_converter, CurrencyConverterError

from advisor_money.distribution import get_distribution_report
from advisor_money.scripts.load_clids import get_clids_mapping

logger = logging.getLogger(__name__)


class BasePartner(object):

    name = ''
    currency = 'USD'
    currency_rate_source = 'ecb'  # 'ecb' or 'cbrf'
    precision = Decimal('10') ** -2
    discovery_clid = None  # clid1011
    shared_revenue = None
    regions = set()

    def __init__(self, report_date):
        self.report_date = report_date
        self._revenue = Decimal()
        self.currency_converter = get_currency_converter(self.currency_rate_source)

    def convert(self, amount, currency, new_currency, date):
        try:
            return self.currency_converter.convert(amount, currency, new_currency, date).quantize(self.precision)
        except CurrencyConverterError:
            raise RuntimeError(traceback.format_exc())

    # date parameter needed here to support function signature because some partners use it
    def rate_date(self, date):
        return self.report_date

    def calculate_revenue(self, revenues, revenue_date):
        currency_rate_date = self.rate_date(revenue_date)

        # filtering revenues using partner's custom logic
        filtered_revenues = filter(self.need_processing, revenues)

        # grouping revenues by currency and summing them up
        grouped_revenues = self.group_revenues(filtered_revenues)
        for currency, revenue in grouped_revenues.iteritems():
            self._revenue += self.shared_revenue * self.convert(
                revenue.quantize(self.precision), currency, self.currency, date=currency_rate_date
            )

    def need_processing(self, revenue):
        return True

    def group_revenues(self, revenues):
        grouped_revenues = defaultdict(Decimal)
        for revenue in revenues:
            currency = revenue['currency']
            grouped_revenues[currency] += self._get_revenue_amount(revenue)
        return grouped_revenues

    def _get_revenue_amount(self, revenue):
        """
        Метод нужен, чтобы посчитать, какую часть выручки учитываем для расчётов с партнёром, т.к. часть
        доходов оседает внутри Яндекса.
        """
        amount = revenue['money']

        # Берём только 50% вала от Директа, т.к. дугие 50% оседают внутри Яндекса
        if revenue['adnetwork'].upper() == 'DIR':
            amount /= 2

        return amount

    @property
    def revenue(self):
        return self._revenue.quantize(self.precision)

    def get_partner_clids(self):
        clids_mapping = get_clids_mapping()
        filtered_clid_pairs = filter(lambda pair: pair[1] == self.discovery_clid, clids_mapping.iteritems())
        clids = map(itemgetter(0), filtered_clid_pairs)
        return clids


class MultiAdsPartner(BasePartner):
    shared_revenue_by_adnetwork = {}

    def group_revenues(self, revenues):
        grouped_revenues = defaultdict(Decimal)
        for revenue in revenues:
            currency = revenue['currency']
            adnetwork = revenue['adnetwork']
            if adnetwork not in self.shared_revenue_by_adnetwork:
                adnetwork = '__default__'
            grouped_revenues[(currency, adnetwork)] += self._get_revenue_amount(revenue)
        return grouped_revenues

    def calculate_revenue(self, revenues, revenue_date):
        currency_rate_date = self.rate_date(revenue_date)

        # filtering revenues using partner's custom logic
        filtered_revenues = filter(self.need_processing, revenues)

        # grouping revenues by currency and summing them up
        grouped_revenues = self.group_revenues(filtered_revenues)

        for key, revenue in grouped_revenues.iteritems():
            currency, adnetwork = key
            self._revenue += self.shared_revenue_by_adnetwork.get(adnetwork, self.shared_revenue) * self.convert(
                revenue.quantize(self.precision), currency, self.currency, date=currency_rate_date
            )


class ActivationsPartner(BasePartner):
    activation_price = None
    activation_price_currency = None
    # list of region ID from geolocation to filter activations
    activation_regions = None

    def __init__(self, report_date):
        super(ActivationsPartner, self).__init__(report_date)
        self._activations_revenue = Decimal()
        self.clids = self.get_partner_clids()
        self.activations = self.get_activations()

    def get_activations(self):
        start_date = self.report_date.replace(day=1)
        end_date = self.report_date

        and_filters = [
            ['clid', 'IN', list(self.clids)],
            ['clid_type_id', '=', '1'],
        ]
        if self.activation_regions:
            and_filters.append(['event_country_id', 'IN', list(self.activation_regions)])

        request = {
            'period': [start_date.isoformat(), end_date.isoformat()],
            'measures': ['activations'],
            'dimensions': ['date|day'],
            'filters': ['AND', and_filters]
        }
        return self._load_activations(request)

    def _load_activations(self, request):
        try:
            activations_json = get_distribution_report(request)
            return self._parse_activations(activations_json)
        except requests.RequestException:
            logger.exception("Can't load activations from distribution API for %s partner", self.name,
                             extra={'data': {'request': request}})
            return {}

    @staticmethod
    def _parse_activations(activations_json):
        activations = {}
        for item in activations_json['data']['report']:
            activations_amount = item['measures'][0]['value']
            activations_date = item['dimensions'][0]['value']
            activations[activations_date] = activations_amount

        return activations

    def calculate_revenue(self, revenues, revenue_date):
        super(ActivationsPartner, self).calculate_revenue(revenues, revenue_date)
        self.calculate_revenue_by_activations(revenue_date)

    def calculate_revenue_by_activations(self, date):
        activations_amount = self.activations.get(date.isoformat())
        currency_rate_date = self.rate_date(date)

        if activations_amount:
            revenue = self.activation_price * Decimal(activations_amount)
            self._activations_revenue += self.convert(revenue, self.activation_price_currency, self.currency, currency_rate_date)

    @property
    def revenue(self):
        revenue = max(self._revenue, self._activations_revenue)
        return revenue.quantize(self.precision)
