import json
import logging

import click
import yt.wrapper as yt

from advisor_money.common.cli import DateType
from advisor_money.settings import YT_CONFIG, CLIDS_JSON_FILE, DISCOVERY_CLID_NUMBER
from advisor_money.utils.date_utils import get_yesterday
from advisor_money.utils.yt_utils import get_yt_path

logger = logging.getLogger(__name__)


def load_clids(date, save_to_file=False):
    table_path = get_yt_path('//statbox/statbox-dict-by-name/distr_report.json/{}'.format(date.isoformat()))

    if not yt.exists(table_path):
        logger.error('Can not load clids. Path {} does not exist.'.format(table_path))

    clids = []
    partners_pack_id = set()
    for row in json.loads(yt.read_file(table_path).read()):
        try:
            clid_info = {
                'clid_number': int(row['clid_type_id']),
                'clid': int(row['id']),
                'set_id': int(row['set_id']),
                'pack_id': int(row['set.pack_id']),
            }
        except ValueError:
            continue

        if clid_info['clid_number'] == DISCOVERY_CLID_NUMBER:
            partners_pack_id.add(clid_info['pack_id'])

        clids.append(clid_info)

    clids = filter(lambda item: item['pack_id'] in partners_pack_id, clids)

    if save_to_file:
        with open(CLIDS_JSON_FILE, 'w') as f:
            json.dump(clids, f, indent=2)

    return clids


def get_clids_mapping():
    try:
        with open(CLIDS_JSON_FILE) as f:
            clids = json.load(f)
    except IOError:
        logger.warning('Can not find json with clids. Running load_clids command.')
        yesterday = get_yesterday()
        clids = load_clids(yesterday, save_to_file=True)

    discovery_clid_by_pack_id = {}
    for clid_info in clids:
        clid = clid_info['clid']
        if clid_info['clid_number'] == DISCOVERY_CLID_NUMBER:
            discovery_clid_by_pack_id[clid_info['pack_id']] = clid

    clids_mapping = {}
    for clid_info in clids:
        clid = clid_info['clid']
        if clid_info['clid_number'] != DISCOVERY_CLID_NUMBER:
            clids_mapping[clid] = discovery_clid_by_pack_id[clid_info['pack_id']]

    return clids_mapping


@click.command()
@click.option('--date', '-d', default=get_yesterday, type=DateType(), help='Date in format yyyy-mm-dd')
def cli(date):
    yt.update_config(YT_CONFIG)

    logger.info("Start downloading clids from distributions tables")
    load_clids(date, save_to_file=True)
    logger.info("Successfully finished downloading clids")


if __name__ == '__main__':
    cli()
