import logging
from collections import Counter
from numbers import Number

import click
import yt.wrapper as yt
from yacurrency import currency_codes, CBRFCurrencyConverter

from advisor_money.common.cli import DateType
from advisor_money.settings import YT_CONFIG, ZEN_MONEY_REPORTS_PATH, LAUNCHER_MONEY_REPORTS_PATH
from advisor_money.utils.date_utils import get_yesterday
from advisor_money.utils.yt_utils import get_yt_path, table_should_exist

logger = logging.getLogger(__name__)

ERROR_CLID_VALUE = -2
TEST_CLID_VALUE = -1

ERROR_CLID_COUNT_THRESHOLD = 0.05
TEST_CLID_MONEY_THRESHOLD = 0.01

converter = CBRFCurrencyConverter()

TABLE_SHOULD_EXIST_TIME = 18  # at 18:00 UTC all reports should already exist


# for more details of validating rules see https://st.yandex-team.ru/ADVISOR-887
# and https://st.yandex-team.ru/ADVISOR-496#1490114860000
def validate_table(table_path):

    if not yt.exists(path=table_path):
        if table_should_exist(TABLE_SHOULD_EXIST_TIME):
            return "2;table {} does not exist".format(table_path)
        else:
            return "0;table {} does not exist yet".format(table_path)

    if table_should_exist(TABLE_SHOULD_EXIST_TIME) and yt.is_empty(table=table_path):
        return "2;table {} is empty".format(table_path)

    invalid_critical_fields_count = Counter()
    invalid_currency_values = set()
    error_clid_count = 0
    test_clid_money = 0
    total_money_sum = 0

    row_count = None
    for row_count, row in enumerate(yt.read_table(table_path, format='json')):
        # validating clid
        try:
            clid = int(row['clid'])
            if clid == ERROR_CLID_VALUE:
                error_clid_count += 1
        except (ValueError, TypeError):
            invalid_critical_fields_count.update(['clid'])
            clid = None

        # validating country_geo_id
        if not isinstance(row['country_geo_id'], int):
            invalid_critical_fields_count.update(['country_geo_id'])

        # validating currency
        currency = row['currency'].upper()
        if currency not in currency_codes:
            invalid_currency_values.add(currency)

        # validating money
        money = row['money']
        if not isinstance(money, Number) or money < 0:
            invalid_critical_fields_count.update(['money'])
        else:
            # trying to convert money to RUB
            try:
                money = float(converter.convert(money, currency))
            except BaseException:  # monrun is crushing cause of some exception in yacurrency module
                pass

            total_money_sum += money
            if clid == TEST_CLID_VALUE:
                test_clid_money += money

    # throw warning if error clids count more then 1%
    if row_count and float(error_clid_count) / row_count > ERROR_CLID_COUNT_THRESHOLD:
        return "1;too many invalid clids: {} of {}".format(error_clid_count, row_count)

    # throw warning if test clid's money more then 1%
    if total_money_sum > 0 and float(test_clid_money) / total_money_sum > TEST_CLID_MONEY_THRESHOLD:
        return "1;invalid currency codes: {}".format(", ".join(invalid_currency_values))
    elif total_money_sum == 0:
        return "1;total money is zero"

    if invalid_currency_values:
        return "2;invalid currency codes: {}".format(", ".join(invalid_currency_values))

    if invalid_critical_fields_count:
        invalid_critical_fields_count_string = '; '.join(
            map(lambda (k, v): '%s - %s times' % (k, v), invalid_critical_fields_count.items())
        )
        return "2;some fields in table {} have invalid values: {}".format(table_path, invalid_critical_fields_count_string)

    return "0;ok"


def validate_reports(date, source):
    if source == 'launcher':
        logger.info("Start validating launcher money reports")
        launcher_report_path = get_yt_path(LAUNCHER_MONEY_REPORTS_PATH.format(date.isoformat()))
        print validate_table(launcher_report_path)
        logger.info("Finished validating launcher money reports")
    elif source == 'zen':
        logger.info("Start validating zen money reports")
        zen_report_path = get_yt_path(ZEN_MONEY_REPORTS_PATH.format(date.isoformat()), root='')
        print validate_table(zen_report_path)
        logger.info("Finished validating zen money reports")
    else:
        logger.error("Unknown source: {}".format(source))


@click.command()
@click.option('--date', '-d', default=get_yesterday, type=DateType(), help='Date in format yyyy-mm-dd')
@click.option('--source', '-s', help='Source of data: launcher or zen')
def cli(date, source):
    if not source:
        logger.error("Parameter source is empty. Chose one: launcher or zen")
        return

    yt.update_config(YT_CONFIG)

    validate_reports(date, source)


if __name__ == '__main__':
    cli()
