#!/usr/bin/env python
import datetime as dt
import logging

import click

from advisor_money.common.cli import DateType
from advisor_money.common.ip_reg_lookup import IpRegLookup
from advisor_money.common.postback_parser import PostbackParser, parse_impression_id, postback_is_testing
from advisor_money.db import advisor_db
from advisor_money.models.postback import Postback
from advisor_money.scripts.parse_postbacks import PROMO_EVENTS_DAYS
from advisor_money.utils.date_utils import get_yesterday, date_to_datetime, date_range

logger = logging.getLogger(__name__)


def match_postbacks_and_promo_events(timestamp):
    logger.info('Parse postbacks from mongo for %s', timestamp.date().isoformat())

    parser = PostbackParser()

    postback_events = advisor_db.postback_events.find({'datetime': {'$gt': timestamp}})
    promo_events = advisor_db.promo_event.find({'timestamp': {'$gt': timestamp - dt.timedelta(days=PROMO_EVENTS_DAYS)}})

    postbacks = {}
    for postback_event in postback_events:
        ip = postback_event['ip']
        request = postback_event['request']

        parsed = parser.parse_postback(request)
        if parsed:
            impression_id = parsed['impression_id']

            if postback_is_testing(impression_id):
                continue

            payout = parsed['payout']
            try:
                payout = float(payout)
            except (ValueError, TypeError):
                payout = None

            postback_info = {
                'adnetwork': parsed['adnetwork'],
                'impression_id': impression_id,
                'payout_unconverted': payout,
                'currency': parsed['currency'],
                'request': request,
                'timestamp': postback_event['datetime'],
                'click_ip': parsed['click_ip'],
                'payout_ip': ip,
            }
            postback_info.update(parse_impression_id(impression_id))

            if impression_id not in postbacks:
                postbacks[impression_id] = postback_info
            else:
                logger.warning("Many postbacks for one impression_id: %s", impression_id)

    has_promo_event = set()
    for promo_event in promo_events:
        impression_id = promo_event['impression_id']
        if promo_event['impression_id'] in postbacks:
            postback = postbacks[impression_id]

            # use info from last promo event only
            if 'click_timestamp' in postback and promo_event['timestamp'] < postback['click_timestamp']:
                continue

            has_promo_event.add(impression_id)
            postback.update(dict(
                    click_timestamp=promo_event['timestamp'],
                    android_id=promo_event['android_id'],
                    country=promo_event['country'],
                    device_id=promo_event['device_id'],
                    uuid=promo_event['uuid'],
                    offer_id=promo_event['offer_id'],
                    place=promo_event['place'],
                    package_name=promo_event['package_name'],
                    view_type=promo_event['view_type'],
                    payout=promo_event['payout']
                ))

            if not postback.get('payout_unconverted'):
                postback['payout_unconverted'] = float(promo_event['payout_unconverted'])

            if not postback.get('currency'):
                postback['currency'] = promo_event['currency']

    bulk = Postback.get_bulk()
    for impression_id, postback_info in postbacks.iteritems():
        if impression_id in has_promo_event:
            Postback(**postback_info).add_to_bulk(bulk)
        else:
            logger.warning("No PromoEvent for non-testing postback: %r", postback_info['request'])

    bulk.execute()


@click.command()
@click.option('--date', '-d', default=get_yesterday, type=DateType(), help='Date in format yyyy-mm-dd')
@click.option('--lookback', '-l', type=int, default=0, help='Number of days ago to run command on')
def cli(date, lookback):
    date = date_to_datetime(date)

    for d in date_range(date - dt.timedelta(days=lookback), date):
        match_postbacks_and_promo_events(d)


if __name__ == '__main__':
    cli()
