import datetime as dt
import logging
from collections import defaultdict
from decimal import Decimal
from itertools import chain

import click
import yt.wrapper as yt

from advisor_money.common.cli import DateType
from advisor_money.partners import all_partners
from advisor_money.scripts.load_clids import get_clids_mapping, load_clids
from advisor_money.settings import (ZEN_MONEY_REPORTS_PATH, LAUNCHER_MONEY_REPORTS_PATH, PARTNER_REPORTS_PATH,
    YT_CONFIG, ISO_DATE_FORMAT)
from advisor_money.utils.date_utils import get_yesterday, date_range
from advisor_money.utils.yt_utils import get_yt_path, make_yt_path, table_should_exist

logger = logging.getLogger(__name__)

ZEN_TABLE_SHOULD_EXIST_TIME = 14  # at 13:00 UTC zen reports should already exist


class PartnerReportUpdater(object):

    def __init__(self, report_date, safe_write=True):
        self.report_date = report_date
        self.partners = {partner.discovery_clid: partner(self.report_date) for partner in all_partners}
        self.clids_mapping = get_clids_mapping()
        self.daily_revenue_diff = defaultdict(list)
        self.safe_write = safe_write

    def run(self):
        self.generate_reports()
        self.write_report()

    def generate_reports(self):
        partner_previous_revenue = defaultdict(Decimal)

        for current_date in self.get_report_dates():
            logger.info('Calculating partners revenue on %s', current_date.isoformat())

            revenues_by_partners = defaultdict(list)
            for revenue in chain(get_zen_data(current_date),
                                 get_launcher_data(current_date)):
                partner_clid = self.clids_mapping.get(revenue['clid'])
                if partner_clid:
                    revenues_by_partners[partner_clid].append(revenue)

            for partner_clid, partner in self.partners.iteritems():
                revenues = revenues_by_partners.get(partner_clid) or []
                partner.calculate_revenue(revenues, current_date)

                # calculating partner's daily revenue diff and saving current revenue
                self.daily_revenue_diff[partner_clid].append(partner.revenue - partner_previous_revenue[partner_clid])
                partner_previous_revenue[partner_clid] = partner.revenue

    def write_report(self):
        for i, current_date in enumerate(self.get_report_dates()):
            rows = []
            for partner_clid, partner in self.partners.iteritems():
                rows.append({
                    'clid': partner_clid,
                    'money': self.daily_revenue_diff[partner_clid][i],
                    'currency': partner.currency,
                    'billing_period': current_date.isoformat(),
                    'name': partner.name,
                })

            report_path = make_yt_path(PARTNER_REPORTS_PATH.format(current_date.isoformat()))
            self.write_table(report_path, rows)

    def write_table(self, path, rows):
        """
        Writes data to YT table.
        If self.safe_write == True, don't overwrite non-zero data with zero values
        """
        if self.safe_write and yt.exists(path):
            old_data = {}
            for row in yt.read_table(path, format='json'):
                clid = row['clid']
                old_data[clid] = row['money']

            for row in rows:
                clid, new_value = row['clid'], row['money']
                old_value = old_data.get(clid, 0)
                if new_value == 0 and old_value != 0:
                    row['money'] = old_value

        yt.write_table(path, rows, format='json')

    def get_report_dates(self):
        """
        Generator of dates from the beginning of month till report_date.
        """
        start_date = self.report_date.replace(day=1)
        end_date = self.report_date
        return date_range(start_date, end_date)


def get_zen_data(report_date):
    table_path = get_yt_path(ZEN_MONEY_REPORTS_PATH.format(report_date.isoformat()))
    return get_postbacks_report_data(table_path, source='zen')


def get_launcher_data(report_date):
    table_path = get_yt_path(LAUNCHER_MONEY_REPORTS_PATH.format(report_date.isoformat()))
    return get_postbacks_report_data(table_path, source='launcher')


def get_postbacks_report_data(report_path, source):

    def get_report_data(path):
        for row in yt.read_table(path, format='json'):
            clid = int(row['clid'])
            money = Decimal(str(row['money']))

            # TODO: validate every field
            if money > 0 and clid > 0:
                yield {
                    'clid': clid,
                    'money': money,
                    'currency': row['currency'],
                    'date': dt.datetime.strptime(row['date'], ISO_DATE_FORMAT),
                    'adnetwork': row['adnetwork'],
                    'country_id': row['country_geo_id'],
                    'source': source,
                }

    if yt.exists(report_path):
        return get_report_data(report_path)
    elif source == 'zen' and not table_should_exist(ZEN_TABLE_SHOULD_EXIST_TIME):
        pass
    else:
        logger.warning('Path {} does not exist'.format(report_path))
    return []


def generate_partner_reports(date, safe_write=True):
    logger.info('Generating partner reports for %s', date.isoformat())
    try:
        updater = PartnerReportUpdater(date, safe_write)
        updater.run()
    except RuntimeError as e:
        logger.error("Failed to generate report.\n%s", e)


@click.command()
@click.option('--date', '-d', default=get_yesterday, type=DateType(), help='Date in format yyyy-mm-dd')
@click.option('--update-clids/--no-update-clids', 'update_clids', default=True, is_flag=True, help='Explicitly update clids')
@click.option('--safe-write/--no-safe-write', 'safe_write', default=True, is_flag=True, help="Don't overwrite non-zero data with zero values")
def cli(date, update_clids, safe_write):
    yt.update_config(YT_CONFIG)

    if update_clids:
        today = dt.date.today()
        logger.info('Loading and saving clids for %s', today.isoformat())
        load_clids(today, save_to_file=True)

    generate_partner_reports(date, safe_write)


if __name__ == '__main__':
    cli()
