import datetime as dt
import logging
from uuid import UUID

import click
import yt.wrapper as yt

from advisor_money.common.cli import DateType
from advisor_money.settings import (YT_CONFIG, LAUNCHER_MONEY_REPORTS_PATH,
                                    ADVISOR_POSTBACKS_PATH)
from advisor_money.utils.date_utils import get_yesterday, date_range
from advisor_money.utils.yt_utils import make_yt_path

logger = logging.getLogger(__name__)


def fetch_device_info(device_id):
    """
    check mongo db for device information (manufacturer and model)
    :param device_id - user's DeviceID:
    :return: dict with MANUFACTURER and MODEL keys
    """
    from advisor_money.db import advisor_replica_db

    device_info = {}
    if not device_id:
        return device_info

    # try to find device in replica
    try:
        device = advisor_replica_db.profile.find_one({'_id': UUID(device_id)})
    except (TypeError, ValueError):
        logger.error('Wrong device_id: %r', device_id)
        return device_info

    if not device:
        logger.info('Device with device_id %s not found', device_id)
        return device_info

    # extract device name from client info
    try:
        fields = device['android_info']['os_build']['string_fields']
    except KeyError:
        return device_info
    if not isinstance(fields, list):
        return device_info
    for field in fields:
        if 'key' in field and field['key'] == 'MANUFACTURER':
            device_info['MANUFACTURER'] = field['value']
        if 'key' in field and field['key'] == 'MODEL':
            device_info['MODEL'] = field['value']
    return device_info


def prepare_report_records(date):
    """
    Prepare rows for partners' report as described in ADVISOR-496.
    For each date new table is created.
    :param self:
    :param date:
    :return:
    """
    from advisor_money.geo import countries_id_map

    postbacks_table = make_yt_path(ADVISOR_POSTBACKS_PATH.format(date.isoformat()))

    for postback in yt.read_table(postbacks_table, format='json'):
        device_info = fetch_device_info(postback.get('device_id'))

        try:
            clid = int(postback['clid'])
        except (ValueError, TypeError):
            clid = -2

        # old adnetworks-postbacks tables don't have 'payout_unconverted' field
        if 'payout_unconverted' in postback:
            money = postback['payout_unconverted']
        else:
            money = postback['payout']

        if money:
            yield {
                'date': str(date),
                'clid': clid,
                'country': postback['country'],
                'country_geo_id': countries_id_map.get(postback['country']),
                'money': money,
                'adnetwork': postback.get('adnetwork', ''),
                'device_manufacturer': device_info.get('MANUFACTURER', '').encode(errors='ignore'),
                'device_model': device_info.get('MODEL', '').encode(errors='ignore'),
                'currency': postback.get('currency', 'USD')
            }


def update_partners_report_yt(date):
    logger.info('Generating postback report for %s', date.isoformat())

    # publish report data
    report_rows = list(prepare_report_records(date))

    if report_rows:
        report_path = make_yt_path(LAUNCHER_MONEY_REPORTS_PATH.format(date.isoformat()))
        yt.write_table(report_path, report_rows, format='json')


@click.command()
@click.option('--date', '-d', default=get_yesterday, type=DateType(), help='Date in format yyyy-mm-dd')
@click.option('--lookback', '-l', type=int, default=0, help='Number of days ago to run command on')
def cli(date, lookback):
    yt.update_config(YT_CONFIG)

    for d in date_range(date - dt.timedelta(days=lookback), date):
        update_partners_report_yt(d)


if __name__ == '__main__':
    cli()
