import datetime as dt
import logging
from collections import defaultdict
from itertools import chain

import click

from advisor_money.common.cli import DateType
from advisor_money.models.postback import Postback
from advisor_money.models.promo_stats import PromoOfferStats
from advisor_money.utils.date_utils import utc_today, date_to_datetime, date_range

logger = logging.getLogger(__name__)


def calculate_promo_stats(report_date):
    logger.info('Calculating promo offer stats for %s', report_date.isoformat())

    if isinstance(report_date, dt.date):
        report_date = date_to_datetime(report_date)

    start_date = report_date
    end_date = start_date + dt.timedelta(days=1)
    daily_stats = Postback.objects(timestamp__gte=start_date, timestamp__lt=end_date).aggregate({
        '$group': {
            '_id': '$offer_id',
            'daily_conversions': {'$sum': 1},
            'daily_payout': {'$sum': '$payout_unconverted'},
        }
    })

    # first day of month
    start_date = dt.datetime(year=report_date.year, month=report_date.month, day=1)
    monthly_stats = Postback.objects(timestamp__gte=start_date, timestamp__lt=end_date).aggregate({
        '$group': {
            '_id': '$offer_id',
            'monthly_conversions': {'$sum': 1},
            'monthly_payout': {'$sum': '$payout_unconverted'},
        }
    })

    offers_stats = defaultdict(dict)
    for item in chain(daily_stats, monthly_stats):
        offer_id = item.pop('_id')
        item['offer_id'] = offer_id
        offers_stats[offer_id].update(item)

    bulk = PromoOfferStats.get_bulk()
    for item in offers_stats.itervalues():
        PromoOfferStats(
            offer_id=item['offer_id'],
            daily_conversions=item.get('daily_conversions', 0),
            monthly_conversions=item.get('monthly_conversions', 0),
            daily_payout=item.get('daily_payout', 0.0),
            monthly_payout=item.get('monthly_payout', 0.0),
            date=report_date,
        ).add_to_bulk(bulk)

    bulk.execute()


@click.command()
@click.option('--date', '-d', default=utc_today, type=DateType(), help='Date in format yyyy-mm-dd')
@click.option('--lookback', '-l', type=int, default=0, help='Number of days ago to run command on')
def cli(date, lookback):
    for d in date_range(date - dt.timedelta(days=lookback), date):
        calculate_promo_stats(d)


if __name__ == '__main__':
    cli()
