from unittest import TestCase
from advisor_money.common.postback_parser import PostbackParser, postback_is_testing


class PostbackParserTest(TestCase):
    def setUp(self):
        self.parser = PostbackParser()

    def test_artofclick(self):
        request = "/api/v1/postback/artofclick?aff_sub=launcher%3Apromo%3B%3A584fc072d299ff53d5ab5692%3A0%3A0%3B0" \
                  "%3B0%3A0%3A0.145.1%3B%3A2247992%3A1.6.9.qa.7003903&payout=0.12&country=RU"

        self.assertDictEqual(self.parser.parse_postback(request), {
            'adnetwork': 'artofclick',
            'impression_id': 'launcher:promo;:584fc072d299ff53d5ab5692:0:0;0;0:0:0.145.1;:2247992:1.6.9.qa.7003903',
            'payout': 0.12,
            'currency': 'USD',
            'click_ip': None,
        })

    def test_pubnative(self):
        request = "/api/v1/postback/pubnative?transaction_id=launcher%3Agoro%3Bgoro%3A585248e88204875e1627c630%3A1%3" \
                  "A0%3B0%3B1%3B0%3B2%3A0%3A0.145.1%3B%3A2247992%3A1.7.0.5002128&offer_id=2247992&cid=102d516142b7bd" \
                  "73462ff8a7c7c2f8&offer_id=37097&revenue=0.18&payout=0.18"
        self.assertDictEqual(self.parser.parse_postback(request), {
            'adnetwork': 'pubnative',
            'impression_id': 'launcher:goro;goro:585248e88204875e1627c630:1:0;0;1;0;2:0:0.145.1;:2247992:1.7.0.5002128',
            'payout': 0.18,
            'currency': 'USD',
            'click_ip': None,
        })

    def test_appnext(self):
        request = "/api/v1/postback/appnext?impressionid=impressionID=launcher:kira;kira:5851754af552d5dbd5d79d20:3:" \
                  "100;50;1;0;2:0:0.145.1;0.10.3:2258705:1.6.9.5002105&clid=2258705&revenue=1.4850&package_name=com." \
                  "playrix.fishdomdd.gplay&did=426CE192-E332-43E7-B6AC-7828F6DD88C8"
        self.assertDictEqual(self.parser.parse_postback(request), {
            'adnetwork': 'appnext',
            'impression_id': 'launcher:kira;kira:5851754af552d5dbd5d79d20:3:100;50;1;0;2:0:0.145.1;'
                             '0.10.3:2258705:1.6.9.5002105',
            'payout': 1.4850,
            'currency': 'USD',
            'click_ip': None,
        })

    def test_supersonic(self):
        request = "/api/v1/postback/supersonic?$51&ImpressionId=launcher%3Ablack%3Bblack%3A585120eef552d5dbcc62027d%" \
                  "3A0%3A0%3B0%3B1%3B0%3B2%3A0%3A0.145.1%3B%3A2271538%3A1.6.7.5002036"
        self.assertDictEqual(self.parser.parse_postback(request), {
            'adnetwork': 'supersonic',
            'impression_id': 'launcher:black;black:585120eef552d5dbcc62027d:0:0;0;1;0;2:0:0.145.1;'
                             ':2271538:1.6.7.5002036',
            'payout': 0.51,
            'currency': 'USD',
            'click_ip': None,
        })

    def test_youappi(self):
        request = "/api/v1/postback/youappi?sub_id=recommendation_widget%3Akira%3Bkira%3A5852da8283ca35bfd1710402%3A" \
                  "4%3A0%3B0%3B1%3B0%3B2%3A0%3A0.152.1%3B%3A2247992%3A1.7.0.5002128%3ARU%3BRU&payout=0.96"
        self.assertDictEqual(self.parser.parse_postback(request), {
            'adnetwork': 'youappi',
            'impression_id': 'recommendation_widget:kira;kira:5852da8283ca35bfd1710402:4:0;0;1;0;2:0:0.152.1;:'
                             '2247992:1.7.0.5002128:RU;RU',
            'payout': 0.96,
            'currency': 'USD',
            'click_ip': None,
        })

    def test_glispa(self):
        request = "/api/v1/postback/glispa?placement=&sid=launcher%3Agoro%3Bgoro%3A5852b8cb1d9806a486730c90%3A3%3A10" \
                  "0%3B50%3B1%3B0%3B2%3A0%3A0.145.1%3B0.10.3%3A2271538%3A1.6.7.5002036&idfa_md5=&idfa_sha1=&androidi" \
                  "d_md5=e94057a65c50c1f13b3ecfb5fd4965de&gaid_md5=958d96a11211e15e620c6c6afe1ae7a2"
        self.assertDictEqual(self.parser.parse_postback(request), {
            'adnetwork': 'glispa',
            'impression_id': 'launcher:goro;goro:5852b8cb1d9806a486730c90:3:100;50;1;0;2:0:0.145.1;0.10.3:2271538:'
                             '1.6.7.5002036',
            'payout': None,  # payout will be matched with click in promo_events
            'currency': 'USD',
            'click_ip': None,
        })

    def test_yeahmobi(self):
        request = "/api/v1/postback/yeahmobi?offer_id=1955598&offer_name=%7Boffer_name%7D&aff_id=102845&aff_sub=laun" \
                  "cher:kitana_dithering;kitana_dithering:58506532d299ff53a8739c75:0:0;0;1;0;2:0:0.145.1;:2247992:1." \
                  "7.0.5002128&aff_sub2=2247992&aff_sub3=&click_id=fc39dc561-c565-1a66-7ca2c009feafb058a6cfe7578379f" \
                  "0095c4bc4331c3001e&click_ip=83.213.111.52&conv_ip=35.166.0.110&datetime=2016-12-13%2021:23:49&ran" \
                  "=%7Bran%7D&payout=0.07&device_brand=Samsung&device_model=SM-J500FN&device_os=6.0"
        self.assertDictEqual(self.parser.parse_postback(request), {
            'adnetwork': 'yeahmobi',
            'impression_id': 'launcher:kitana_dithering;kitana_dithering:58506532d299ff53a8739c75:0:0;0;1;0;2:0:'
                             '0.145.1;:2247992:1.7.0.5002128',
            'payout': 0.07,
            'currency': 'USD',
            'click_ip': '83.213.111.52',
        })

    def test_cheetah(self):
        request = "/api/v1/postback/cheetah?impression_id=launcher:subzero_promo;promo:59923c36f16056bd1b0864f7:1" \
                  "&sub2=;1;1:0;0;0:0:1.22.0;1.26.3:2246894:1.6.6.500&sub3=2027:RU;RU&sub4=&payout=2.4"
        self.assertDictEqual(self.parser.parse_postback(request), {
            'adnetwork': 'cheetah',
            'impression_id': 'launcher:subzero_promo;promo:59923c36f16056bd1b0864f7:1;1;1:0;0;0:0:1.22.0;1.26.3:'
                             '2246894:1.6.6.5002027:RU;RU',
            'payout': 2.4,
            'currency': 'USD',
            'click_ip': None,
        })

    def test_direct(self):
        request = "/api/v1/postback/direct-plarium?impression_id=recommendation_widget%3Asubzero_promo%" \
                  "3Bpromo%3A59930934a819e79afdd99da8%3A1%3B1%3B4%3A0%3B0%3B0%3A0%3A1.22.0%3B1.26.4%3A2246894%3A1." \
                  "6.6.5002027%3ARU%3BRU&payout=0.70&currency=USD"
        self.assertDictEqual(self.parser.parse_postback(request), {
            'adnetwork': 'direct-plarium',
            'impression_id': 'recommendation_widget:subzero_promo;promo:59930934a819e79afdd99da8:1;1;4:0;0;0:0:'
                             '1.22.0;1.26.4:2246894:1.6.6.5002027:RU;RU',
            'payout': 0.7,
            'currency': 'USD',
            'click_ip': None,
        })

    def test_parse_postback_request_params_with_empty_params(self):
        request = "/api/v1/postback/ironsource?"
        self.assertIsNone(self.parser.parse_postback_request_params(request))

    def test_parse_postback_request_params_with_empty_adnetwork_name(self):
        request = "/api/v1/postback/"
        self.assertIsNone(self.parser.parse_postback_request_params(request))

    def test_postback_is_testing(self):
        impression_id = 'recommendation_widget:subzero_promo;promo:59930934a819e79afdd99da8:1;1;4:0;0;0:0:1.22.0;1.26.4:2246894:%s:RU;RU'
        self.assertFalse(postback_is_testing(impression_id % '1.5.2.1000133'))
        self.assertTrue(postback_is_testing(impression_id % '1.5.2.1000133qa'))
        self.assertTrue(postback_is_testing(impression_id % '1.5.2.1000133dev'))
