import logging
from calendar import timegm
from distutils.version import LooseVersion
from time import sleep
from traceback import format_stack

import six
from _shared_localization import UserInfo, getCache
from django.conf import settings

logger = logging.getLogger(__name__)

DEVICE_PHONE = 'phone'
DEVICE_TABLET = 'tablet'


class LazyItemsCache(dict):
    def __getitem__(self, project):
        if project not in self:
            self._load(project)
        return super(LazyItemsCache, self).__getitem__(project)

    def _load(self, project):
        config_path = settings.LOCALIZATION_CONFIG_PATH
        mongo_uri = settings.LOCALIZATION_MONGO_URI
        db_name = settings.LOCALIZATION_MONGO_DBNAME

        cache = getCache(project, config_path, mongo_uri, db_name)
        while not cache.isReady():
            sleep(0.05)

        self[project] = cache


_projects_items_cache = LazyItemsCache()


class ItemNotFound(RuntimeError):
    pass


class LocalizationItem(object):
    def __init__(self, project, user_info, name):
        self.project = project
        self.user_info = user_info
        self.name = str(name)
        self._value = None

    @property
    def value(self):
        if self._value is None:
            try:
                value = _projects_items_cache[self.project].getItemValue(self.name, self.user_info)
                self._value = value.decode('utf-8') if isinstance(value, six.binary_type) else value
            except RuntimeError:
                raise ItemNotFound('Item %r not found.' % self.name)

        return self._value

    @property
    def expire_time(self):
        return _projects_items_cache[self.project].getExpirationDate(self.name, self.user_info)


@six.python_2_unicode_compatible
class LocalizationUser(object):
    def __init__(self, uuid=None, language=None, app_name=None, app_version=None, created_at=None, clids=None,
                 country=None, region_ids=None, region_ids_init=None, loyalty=None, device_type=DEVICE_PHONE,
                 device_model=None, device_manufacturer=None, os_name=None, os_version=None,
                 serial_number=None, yphone_id=None, yphone_batch=None):
        """
        :param uuid: user's UUID
        :param language: ISO 639-1 2-letter language code
        :param app_name: application name
        :param app_version: application version
        :param created_at: user creation time (datetime object)
        :param clids: dict of user's clids {clid number: integer clid}
        :param country: user's ISO 3166 country code
        :param region_ids: user's current location geo code
        :param region_ids_init: user's geo code array
        :param loyalty: user's loyalty from crypta
        :param device_type: type of a device (e.g. phone, tablet). Default is phone.
        :param device_model: mobile device model
        :param device_manufacturer: mobile device manufacturer
        :param os_name: user's operating system name
        :param os_version: user's operating system version
        :param serial_number: device serial number
        :param yphone_id: Yandex Phone ID
        """
        self.uuid = uuid
        self.language = language
        self.app_name = app_name
        self.app_version = app_version
        self.created_at = created_at
        self.clids = clids
        self.country = country
        self.region_ids = region_ids
        self.region_ids_init = region_ids_init
        self.loyalty = loyalty
        self.device_type = device_type
        self.device_model = device_model
        self.device_manufacturer = device_manufacturer
        self.os_name = os_name
        self.os_version = os_version
        self.serial_number = serial_number
        self.yphone_id = yphone_id
        self.yphone_batch = yphone_batch

    def __str__(self):
        fields = ((k, v) for (k, v) in six.iteritems(self.__dict__) if v is not None)
        return u'LocalizationUser(%s)' % u', '.join((u'%s="%s"' % (key, value) for key, value in fields))


def in_serialized_list(user_value, config_value):
    return any(user_value.lower() == value.lower() for value in config_value.split('|'))


def _create_user_info(user):
    user_info = UserInfo()
    if user.uuid:
        user_info.setUuid(user.uuid.hex)

    if user.language and user.country:
        user_info.setLocale(str(user.language), str(user.country))

    if user.app_name is not None and user.app_version is not None:
        user_info.setApplication(user.app_name, str(user.app_version))

    if user.region_ids:
        user_info.setRegionIds(user.region_ids)

    if user.region_ids_init:
        user_info.setRegionIdsInit(user.region_ids_init)

    if user.created_at:
        user_created_at_string = str(int(timegm(user.created_at.timetuple())))
        user_info.setExtendedParam(
            'user_creation_date', user_created_at_string,
            lambda user_value, config_value: int(user_value) > int(config_value)
        )
        user_info.setExtendedParam(
            'user_creation_date_to', user_created_at_string,
            lambda user_value, config_value: int(user_value) < int(config_value)
        )

    if user.clids is not None:
        for clid_number, clid_value in six.iteritems(user.clids):
            try:
                user_info.setClid(int(clid_number), int(clid_value))
            except ValueError:
                logger.warning('Invalid value of clid %r: %r', clid_number, clid_value)

    if user.loyalty:
        user_info.setExtendedParam(
            'crypta_loyalty', str(user.loyalty),
            lambda user_value, config_value: float(user_value) > float(config_value)
        )

    if user.device_type:
        user_info.setDeviceType(user.device_type.encode('utf-8'))

    if user.device_model and user.device_manufacturer:
        user_info.setModel(user.device_manufacturer.encode('utf-8'), user.device_model.encode('utf-8'))

    if user.os_name is not None:
        user_info.setExtendedParam(
            'os_name', user.os_name,
            lambda user_value, config_value: user_value.lower() == config_value.lower()
        )

    if user.os_version is not None:
        user_info.setExtendedParam(
            'min_os_version', user.os_version,
            lambda user_value, config_value: LooseVersion(user_value) >= LooseVersion(config_value)
        )
        user_info.setExtendedParam(
            'max_os_version', user.os_version,
            lambda user_value, config_value: LooseVersion(user_value) <= LooseVersion(config_value)
        )

    if user.serial_number is not None:
        user_info.setExtendedParam(
            'serial_number', user.serial_number,
            in_serialized_list
        )

    if user.yphone_id is not None:
        user_info.setExtendedParam(
            'yphone_id', user.yphone_id,
            in_serialized_list
        )

    if user.yphone_batch is not None:
        user_info.setExtendedParam(
            'yphone_batch', user.yphone_batch,
            in_serialized_list
        )

    return user_info


class UserSpecificConfig(object):
    def __init__(self, user, project):
        self.user = user
        self.project = project

    def get_value(self, param_name, default_value=None, log_missing=True):
        item = self.get_item(param_name)
        try:
            return item.value
        except ItemNotFound as e:
            if log_missing:
                stack = format_stack(limit=5)[:-2]
                text = u'{message}. {user}.\ncalls stack (most recent call last):\n{stack}'
                logger.warning(text.format(message=e, stack=''.join(stack), user=self.user), extra={'sample_rate': 0.1})
        return default_value

    def get_item(self, param_name):
        return LocalizationItem(
            self.project,
            self.get_user_info(),
            param_name
        )

    def get_user_info(self):
        return _create_user_info(self.user)

    def get_all_enabled_items(self):
        user_info = self.get_user_info()

        all_enabled_items = []
        for name in _projects_items_cache[self.project].getAllEnabledItems(user_info):
            item = LocalizationItem(self.project, user_info, name)
            if item.value is not None:
                all_enabled_items.append(item)

        return all_enabled_items


class Translator(UserSpecificConfig):
    def _swap_locale(self, language, country):
        old_values = self.user.language, self.user.country
        self.user.language, self.user.country = language, country
        return old_values

    def _set_default_locale(self):
        self.user.language = 'en'
        self.user.country = 'US'

    def translate(self, key):
        result = self.get_value(key, default_value=None)
        if result is not None:
            result.language = self.user.language
            return result

        old_values = self._swap_locale('en', 'US')
        result = self.get_value(key)
        if result is not None:
            result.language = self.user.language
        self._swap_locale(*old_values)
        return result
