import json
import logging
import re

import requests
from flask import Flask, request, jsonify

# https://doc.qloud.yandex-team.ru/doc/api/environment#overviewvers
QLOUD_ENVIRONMENT_OVERVIEW_URL = 'https://platform.yandex-team.ru/api/v1/environment/overview/{environment_id}/{version}'
QLOUD_ENVIRONMENT_URL_TEMPLATE = 'https://platform.yandex-team.ru/projects/{}'

app = Flask(__name__)
app.config.from_object('settings')

logger = logging.getLogger(__name__)


@app.route("/qloud", methods=['POST'])
def qloud_deploy_hook():
    # Check if we should commit
    msg = request.json
    logger.info(json.dumps(msg))
    if msg['status'] not in app.config['QLOUD_NOTIFICATION_STATUSES']:
        return jsonify({'status': 'ok'})
    qloud_application = msg['environmentId'].split('.')[1]
    try:
        url = app.config['SLACK_HOOKS'][qloud_application]
    except KeyError:
        return jsonify({'error': 'Unrecognized qloud application: {}'.format(qloud_application)}), 404

    text = '*{environment}* {status}.'.format(environment=msg['environmentId'].split('.')[-1],
                                              status=msg['status'])

    # Add component versions
    component_versions = get_component_versions(msg['environmentId'], msg['version'])
    if component_versions:
        components_text = '\n'.join('{}={}'.format(name, version) for name, version in component_versions)
        text += "\nComponent versions:\n{components}".format(components=components_text)

    # Add Qloud button
    q_url = QLOUD_ENVIRONMENT_URL_TEMPLATE.format('/'.join(msg['environmentId'].split('.')))
    qloud_button = {
        'type': 'button',
        'text': ':platform: Qloud',
        'url': q_url
    }
    attachment = {
        'fallback': 'Links',
        'actions': [qloud_button]
    }

    # Add Arcanum button and commit message
    version = re.search(':(\d{7,10})[,}].*$', msg['comment'])
    if version:
        a_url = 'https://a.yandex-team.ru/arc/commit/{}'.format(version.groups()[0])

        attachment['actions'].append({
            'type': 'button',
            'text': ':arcanum: Arcanum',
            'url': a_url
        })

        commit = get_commit_message(version.groups()[0])
        text += '\n*{author}*: _"{message}"_'.format(author=commit['author'],
                                                     message=commit['message'].split('\n')[0])
        # Add ST link
        ticket = re.search('[A-Z]{3,}-\d+', commit['message'])
        if ticket:
            ticket = ticket.group()
            st_url = 'https://st.yandex-team.ru/{}'.format(ticket)

            attachment['actions'].append({
                'type': 'button',
                'text': ':st: %s' % ticket,
                'url': st_url
            })
    else:
        text += "\nDeploy comment: {comment}".format(comment=msg['comment'])

    # Notify
    requests.post(url, json={'text': text, 'attachments': [attachment]})

    return jsonify({'status': 'ok'})


def get_component_versions(environment_id, environment_version):
    url = QLOUD_ENVIRONMENT_OVERVIEW_URL.format(environment_id=environment_id, version=environment_version)
    response = requests.get(url, headers={'Authorization': 'OAuth {}'.format(app.config['QLOUD_OAUTH_TOKEN'])})
    component_versions = []
    if response.ok:
        environment = response.json()
        for component_name, component in environment['components'].iteritems():
            if component['type'] == 'standard':
                _, _, component_version = component['repository'].rpartition(':')
                component_versions.append((component_name, component_version))
    return component_versions


def get_commit_message(commit_id):
    try:
        logger.info('Trying to get commit message')
        response = requests.get('http://a.yandex-team.ru/api/tree/commit/%s' % commit_id,
                                headers={'Authorization': 'OAuth %s' % app.config['ROBOT_ARCANUM_TOKEN']})
        if response.ok:
            response = response.json()
            return response
    except requests.ConnectionError:
        pass
    return ''
