#include "config.hpp"

#include <map>
#include <pwd.h>
#include <sys/stat.h>
#include <unistd.h>

#include <boost/algorithm/string.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/property_tree/ini_parser.hpp>


namespace shared_localization
{

namespace
{

inline bool file_exists(const std::string& path) // also returns false if not enough permissions to read directory
{
    struct stat buffer;
    return (stat(path.c_str(), &buffer) == 0);
}

std::string get_username()
{
    struct passwd* pws;
    pws = getpwuid(geteuid());
    if (pws == nullptr)
    {
        throw std::runtime_error("Unable to determine user");
    }
    return pws->pw_name;
}

std::string get_real_path(const std::string& path)
{
    boost::filesystem::path resolved_path(path);
    return boost::filesystem::absolute(resolved_path).string();
}

template <typename T>
T get(const boost::property_tree::ptree& pt, const std::string& key)
{
    try
    {
        return pt.get<T>(key);
    }
    catch(...)
    {
        throw MalformedConfigError("Unable to retrieve key " + key + " from config");
    }
}

template <typename T>
T get(const boost::property_tree::ptree& pt, const std::string& key, const T& default_value)
{
    try
    {
        return pt.get<T>(key, default_value);
    }
    catch(...)
    {
        throw MalformedConfigError("Unable to retrieve key " + key + " from config");
    }
}

}  // end of anonymous namespace

MalformedConfigError::MalformedConfigError(const std::string& reason)
    : std::logic_error(reason)
{}

const std::string Config::run_directory = "/var/tmp";
const std::string Config::tmp_directory = "/tmp";

Config::Config()
{}

Config::Config(const std::string& path)
{
    loadFromFile(path);
}

void Config::loadFromFile(const std::string& path)
{
    std::ifstream infile(path);
    if (!infile.good())
    {
        throw MalformedConfigError("Unable to load config from file " + path);
    }

    boost::property_tree::ptree pt;
    boost::property_tree::ini_parser::read_ini(path, pt);

    // Username
    auto linux_username = get_username();

    // Mongo options
    mongo_retry_timeout = get<unsigned int>(pt, "MongoOptions.retry_timeout_microseconds");

    // SHM options
    memory_segment_name = get<std::string>(pt, "SHMOptions.memory_segment_name",
        "shared_localizations_" + project_name + "_" + linux_username);
    memory_limit = get<unsigned long long>(pt, "SHMOptions.memory_limit");
    bucket_number = get<unsigned int>(pt, "SHMOptions.bucket_number");

    // Logging options
    logfile_path = get<std::string>(pt, "LoggingOptions.log_file",
        "/var/log/yandex/shared_localizations/" + project_name + "-" + linux_username + ".log");
    log_format = get<std::string>(pt, "LoggingOptions.log_format");
    log_level = get<unsigned int>(pt, "LoggingOptions.log_level");

    // Sync options
    pidfile_name = get<std::string>(pt, "SyncOptions.pid_file",
        "shared_localizations_" + project_name + "-" + linux_username + ".pid");
    manager_check_count = get<std::uint64_t>(pt, "SyncOptions.manager_check_count", 1000);

    // Content options
    applications_collection_subname = get<std::string>(pt, "ContentOptions.applications_collection_subname");
    update_interval = get<unsigned int>(pt, "ContentOptions.update_interval_seconds");
}

void Config::validate() const
{
    // Logfile
    if (!file_exists(logfile_path))
    {
        std::ofstream logfile(logfile_path);
        chmod(logfile_path.c_str(), 0644);
    }
    std::ofstream logfile(logfile_path, std::ios_base::app);
    if (!logfile.good())
        throw std::runtime_error("Unable to open logfile " + logfile_path);
}

std::string Config::getCollectionName(const std::string& collection_subname) const
{
    return "localization." + collection_subname;
}

std::string Config::getPidfilePath() const
{
    return Config::run_directory + "/" + pidfile_name;
}

std::string Config::getLockfilePath() const
{
    return Config::tmp_directory + "/" + pidfile_name + ".lock";
}

} // end of shared_localization namespace
