#include <Python.h>
#include <datetime.h>
#include <vector>
#include <string>

#include <boost/date_time/posix_time/posix_time.hpp>


template <typename T>
struct type_into_python
{
    static PyObject* convert(const T&);
};

template <>
PyObject* type_into_python<std::string>::convert(const std::string& t)
{
    return PyUnicode_FromString(t.c_str());
}

template <>
PyObject* type_into_python<boost::posix_time::ptime>::convert(const boost::posix_time::ptime& t)
{
    try
    {
        boost::gregorian::date d = t.date();
        boost::posix_time::time_duration tod = t.time_of_day();
        long usec = tod.total_microseconds() % 1000000;
        return PyDateTime_FromDateAndTime(d.year(), d.month(), d.day(), tod.hours(), tod.minutes(), tod.seconds(), usec);
    }
    catch(...)
    {
        Py_INCREF(Py_None);
        return Py_None;
    }
}


template <>
PyObject* type_into_python<std::vector<std::string>>::convert(const std::vector<std::string>& t)
{
    PyObject* result = PyList_New(0);

    for (auto it = t.cbegin(); it != t.cend(); ++it)
        PyList_Append(result, type_into_python<std::string>::convert(*it));

    return result;
}


class SetExtendedParamWrapper
{
public:
    explicit SetExtendedParamWrapper(const boost::python::object& wrapped)
        : wrapped_(wrapped)
    {}
    bool operator()(const std::string& name, const std::string& value)
    {
        return boost::python::extract<bool>(wrapped_(name.c_str(), value.c_str()));
    }
private:
    boost::python::object wrapped_;
};

struct boost_function_from_python_function
{
    typedef boost::function<bool (const std::string& name, const std::string& value)> BoostFunctionType;

    boost_function_from_python_function()
    {
        boost::python::converter::registry::push_back(
            &convertible,
            &construct,
            boost::python::type_id<BoostFunctionType>());
    }

    static void* convertible(PyObject* obj_ptr)
    {
        if (!PyCallable_Check(obj_ptr))
            return 0;
        return obj_ptr;
    }

    static void construct(PyObject* obj_ptr, boost::python::converter::rvalue_from_python_stage1_data* data)
    {
        boost::python::handle<> handle(boost::python::borrowed(obj_ptr));
        typedef boost::python::converter::rvalue_from_python_storage<BoostFunctionType> StorageType;
        void* storage = reinterpret_cast<StorageType*>(data)->storage.bytes;
        boost::python::object func(handle);
        data->convertible = new(storage) BoostFunctionType(SetExtendedParamWrapper(func));
    }
};
