#include "daemon.hpp"

#include <errno.h>
#include <fstream>
#include <unistd.h>
#include <signal.h>
#include <sys/file.h>

#include <boost/lexical_cast.hpp>

#include <Poco/Logger.h>
#include <Poco/LogStream.h>


namespace
{

struct ScopedFileLock
{
    explicit ScopedFileLock(const std::string& filename)
        : filename_(filename)
    {
        fd_ = open(filename_.c_str(), O_RDONLY | O_CREAT, S_IRUSR | S_IWUSR);
        lock_success_ = ! flock(fd_, LOCK_EX | LOCK_NB);
    }
    ~ScopedFileLock()
    {
        if (lock_success_)
            flock(fd_, LOCK_UN);
        close(fd_);
        unlink(filename_.c_str());
    }
    bool successfullyLocked() const
    {
        return lock_success_;
    }
private:
    bool lock_success_;
    const std::string filename_;
    int fd_;
};

}  // end of anonymous namespace

void set_pid_file(const std::string& lockfile_path, const std::string& pidfile_path)
{
    Poco::LogStream log_stream(Poco::Logger::get("SLManager"));
    // Acquire lock to prevent race conditions
    ScopedFileLock lock(lockfile_path);
    if (!lock.successfullyLocked())
    {
        log_stream.fatal() << "Unable to acquire lock " << lockfile_path << ". Exiting" << std::endl;
        throw std::runtime_error("Unable to acquire lock for pidfile!");
    }

    // Check if pidfile exists and contains pid of running process
    std::ifstream existing_pid_file(pidfile_path);
    if (existing_pid_file.good())
    {
        try
        {
            std::string str;
            std::getline(existing_pid_file, str);
            pid_t pid = boost::lexical_cast<pid_t>(str);
            if ( (kill(pid, 0) == 0 ) || (errno != ESRCH) )
            {
                log_stream.fatal() << "Pid " << pid << " from pidfile " << pidfile_path
                    << " is valid and running. Exiting" << std::endl;
                throw std::runtime_error("Pid from pidfile is valid and running! Exiting");
            }
            else
            {
                log_stream.warning() << "Pid " << pid << " from pidfile " << pidfile_path
                    << " is not running. Overriding" << std::endl;
                std::remove(pidfile_path.c_str());
            }
        }
        catch(const boost::bad_lexical_cast&)
        {
            log_stream.warning() << "Pidfile " << pidfile_path << "does not contain valid pid, overriding." << std::endl;
            std::remove(pidfile_path.c_str());
        }
    }

    // Write our pid to pidfile and start managing
    std::ofstream pid_file(pidfile_path, std::fstream::out);  // 644 by default
    if (!pid_file.good())
    {
        log_stream.fatal() << "Unable to create pidfile " << pidfile_path << ", exiting" << std::endl;
        throw std::runtime_error("Unable to create pidfile!");
    }
    pid_file << getpid() << std::endl;
    pid_file.close();
}

void pidfile_cleanup(const std::string& pidfile_path)
{
    std::remove(pidfile_path.c_str());
}
