#include "items_cache.hpp"

#include "config.hpp"

#include <fstream>
#include <unistd.h>
#include <signal.h>
#include <stdexcept>
#include <sys/types.h>

#include <boost/interprocess/managed_shared_memory.hpp>
#include <boost/interprocess/sync/sharable_lock.hpp>
#include <boost/lexical_cast.hpp>


#ifndef SHARED_LOCALIZATION_VERSION
#define SHARED_LOCALIZATION_VERSION 1
#endif

namespace shared_localization
{

namespace
{

void start_manager(const std::string& executable_path, const Config& config, const std::string& config_path,
                   const std::string& mongo_uri, const std::string& db_name)
{
    pid_t pid = fork();
    if (pid < 0)  // Failed to fork
    {
        throw std::runtime_error("Unable to fork manager");
    }
    else if (pid == 0)  // Child
    {
        execl(executable_path.c_str(), "SLManager", config.project_name.c_str(), config_path.c_str(),
              mongo_uri.c_str(),  db_name.c_str(), (char *) 0);
        // If we got here, it means execl failed
        std::cerr << "Unable to execute manager! Errno: " << errno << std::endl;
        _exit(EXIT_FAILURE);
    }
}

pid_t get_manager_pid(const std::string& pidfile_path)
{
    std::ifstream pid_file(pidfile_path);
    if (pid_file.good())  // exists
    {
        try
        {
            std::string str;
            std::getline(pid_file, str);
            int pid = boost::lexical_cast<pid_t>(str);
            if ( (kill(pid, 0) != 0) && (errno == ESRCH) )
            {
                return 0;
            }
            return pid;
        }
        catch(const boost::bad_lexical_cast&)
        {
            return 0;
        }
    }
    else
    {
        return 0;
    }
}

class StringComparator
{
public:
    bool operator()(const std::string& lhs, const ShmString& rhs) const
    {
        return lhs == std::string(rhs.data(), rhs.size());
    }
};

}  // end of anonymous namespace


std::map<std::string, std::weak_ptr<ItemsCache>> ItemsCache::instances_;
std::string ItemsCache::executable_path = "/usr/bin/SLManager";

ItemsCacheProxy ItemsCache::getCache(const std::string& project_name, const std::string& config_path,
                                     const std::string& mongo_uri, const std::string& db_name)
{
    auto it = instances_.find(project_name);
    if (it != instances_.end())
    {
        if (it->second.expired())
            instances_.erase(it);
        else
            return ItemsCacheProxy(it->second.lock());
    }
    std::shared_ptr<ItemsCache> shared(new ItemsCache(project_name, config_path, mongo_uri, db_name));
    instances_.emplace(std::piecewise_construct,
        std::forward_as_tuple(project_name),
        std::forward_as_tuple(shared));
    return ItemsCacheProxy(shared);
}

bool ItemsCache::allAreReady()
{
    for (const auto& cache: instances_)
    {
        if ((!cache.second.expired()) && (!cache.second.lock()->isReady()))
            return false;
    }
    return true;
}

void ItemsCache::setExecutablePath(const std::string& path)
{
    executable_path = path;
}

ItemsCache::ItemsCache(const std::string& project_name, const std::string& config_path,
                       const std::string& mongo_uri, const std::string& db_name)
    : segment_(nullptr)
{
    config_.project_name = project_name;
    config_.loadFromFile(config_path);
    config_.validate();
    if(!get_manager_pid(config_.getPidfilePath()))
    {
        start_manager(executable_path, config_, config_path, mongo_uri, db_name);
    }
    tryGetSharedContainer();
}

bool ItemsCache::tryGetSharedContainer() const
{
    if (!segment_)
    {
        try
        {
            segment_.reset(
                new boost::interprocess::managed_shared_memory(
                    boost::interprocess::open_only,
                    config_.memory_segment_name.c_str()));
        }
        catch (const boost::interprocess::interprocess_exception& e)
        {
            return false;
        }
    }
    if (!container_)
    {
        try
        {
            container_ = segment_->find<LocalItemsContainer>("container").first;
        }
        catch (const boost::interprocess::interprocess_exception& e)
        {
            return false;
        }
        // Container acquired, perform checks
        if (get_manager_pid(config_.getPidfilePath()) != container_.get()->manager_pid)
        {
            segment_.reset();
            container_.reset();
            return false;
        }
        if (container_.get()->manager_version != SHARED_LOCALIZATION_VERSION)
        {
            throw std::runtime_error("Active manager's version is different from expected. Unable to proceed.");
        }
        boost::interprocess::scoped_lock<boost::interprocess::interprocess_sharable_mutex> lock(container_.get()->clients_mutex);
        container_.get()->clients.insert(getpid());
    }
    return true;
}

ItemsCache::~ItemsCache()
{
    try
    {
        if (isReady())
        {
            boost::interprocess::scoped_lock<boost::interprocess::interprocess_sharable_mutex> lock(container_.get()->clients_mutex);
            container_.get()->clients.erase(getpid());
        }
        auto manager_pid = get_manager_pid(config_.getPidfilePath());
        if (manager_pid)
            kill(manager_pid, SIGUSR1);
    }
    catch(...)
    {}
}

bool ItemsCache::isReady() const
{
    if ((!segment_) || (!container_))
    {
        return tryGetSharedContainer() && container_.get()->is_ready;
    }
    if (config_.manager_check_count && (++requests_after_last_manager_check > config_.manager_check_count))
    {
        checkManagerStatus();
    }
    return container_.get()->is_ready;
}

void ItemsCache::checkManagerStatus() const
{
    auto active_manager_pid = get_manager_pid(config_.getPidfilePath());
    if (active_manager_pid == container_.get()->manager_pid)   // OK
    {
        requests_after_last_manager_check = 0;
        return;
    }
    // We are connected to dead / not actual manager
    if (active_manager_pid)  // Try to seamlessly reacquire cache
    {
        try
        {
            std::unique_ptr<boost::interprocess::managed_shared_memory> segment;
            segment.reset(
                  new boost::interprocess::managed_shared_memory(
                      boost::interprocess::open_only,
                      config_.memory_segment_name.c_str()));
            boost::optional<LocalItemsContainer *> container = segment->find<LocalItemsContainer>("container").first;
            if ((container.get()->manager_version == SHARED_LOCALIZATION_VERSION) &&
                (container.get()->manager_pid == active_manager_pid) &&
                (container.get()->is_ready))
            {
                // Not thread-safe
                segment_.swap(segment);
                container_.swap(container);
                requests_after_last_manager_check = 0;
                return;
            }
        }
        catch (const boost::interprocess::interprocess_exception&)
        {}
    }
    throw std::runtime_error("Localizations error: manager is dead and can't be reacquired.");
}

bool ItemsCache::isItemEnabled(const std::string& item_name, const UserInfo& user_info) const
{
    if (!isReady())
        throw std::runtime_error("Localizations are not ready");
    boost::interprocess::sharable_lock<boost::interprocess::interprocess_sharable_mutex> lock(container_.get()->cache_mutex);
    // implicitly using that boost::hash give same result on equal std::string and ShmString
    auto it = container_.get()->cache.find(item_name, boost::hash<std::string>(), StringComparator());
    if (it == container_.get()->cache.end())
    {
        return false;
    }
    const auto& item_options = it->second.first;
    const auto& item_localizations = it->second.second;
    for (const auto& item_localization: item_localizations)
    {
        if (user_info.matchesConditions(item_localization.conditions, item_options))
            return true;
    }
    return false;
}

std::vector<std::string> ItemsCache::getAllItems() const
{
    if (!isReady())
        throw std::runtime_error("Localizations are not ready");

    std::vector<std::string> result;
    boost::interprocess::sharable_lock<boost::interprocess::interprocess_sharable_mutex> lock(container_.get()->cache_mutex);
    for (auto it = container_.get()->cache.cbegin(); it != container_.get()->cache.cend(); ++it)
    {
        result.push_back(std::string(it->first.data(), it->first.size()));
    }
    return result;
}

std::vector<std::string> ItemsCache::getAllEnabledItems(const UserInfo& user_info) const
{
    auto names =  getAllItems();
    std::vector<std::string> result;
    for (const auto& name: names)
    {
        if (isItemEnabled(name, user_info))
            result.push_back(name);
    }
    return result;
}

std::string ItemsCache::getItemValue(const std::string& item_name, const UserInfo& user_info) const
{
    if (!isReady())
        throw std::runtime_error("Localizations are not ready");
    boost::interprocess::sharable_lock<boost::interprocess::interprocess_sharable_mutex> lock(container_.get()->cache_mutex);
    // implicitly using that boost::hash give same result on equal std::string and ShmString
    auto it = container_.get()->cache.find(item_name, boost::hash<std::string>(), StringComparator());
    if (it == container_.get()->cache.end())
    {
        throw std::runtime_error("No such item!");
    }
    const auto& item_options = it->second.first;
    const auto& item_localizations = it->second.second;
    for (const auto& item_localization: item_localizations)
    {
        if (user_info.matchesConditions(item_localization.conditions, item_options))
            return item_localization.value
                ? std::string(item_localization.value.get().data(), item_localization.value.get().size())
                : "";
    }
    throw std::runtime_error("Item is not enabled!");
}

boost::posix_time::ptime ItemsCache::getExpirationDate(const std::string& item_name, const UserInfo& user_info) const
{
    if (!isReady())
        throw std::runtime_error("Localizations are not ready");
    boost::interprocess::sharable_lock<boost::interprocess::interprocess_sharable_mutex> lock(container_.get()->cache_mutex);
    // implicitly using that boost::hash give same result on equal std::string and ShmString
    auto it = container_.get()->cache.find(item_name, boost::hash<std::string>(), StringComparator());
    boost::posix_time::ptime result = boost::posix_time::neg_infin;
    if (it == container_.get()->cache.end())
    {
        return result;
    }
    const auto& item_options = it->second.first;
    const auto& item_localizations = it->second.second;
    for (const auto& item_localization: item_localizations)
    {
        if (user_info.matchesConditions(item_localization.conditions, item_options))
        {
            if (item_localization.conditions.time_end > result)
                result = item_localization.conditions.time_end;
        }
    }
    return result;
}


ItemsCacheProxy::ItemsCacheProxy(const std::shared_ptr<ItemsCache>& ptr)
    : cache_(ptr)
{}

ItemsCacheProxy::ItemsCacheProxy(const ItemsCacheProxy& rhs)
    : cache_(rhs.cache_)
{}

ItemsCacheProxy::ItemsCacheProxy(ItemsCacheProxy&& rhs)
    : cache_(std::move(rhs.cache_))
{}

bool ItemsCacheProxy::isReady() const
{
    return cache_->isReady();
}

bool ItemsCacheProxy::isItemEnabled(const std::string& item_name, const UserInfo& user_info) const
{
    return cache_->isItemEnabled(item_name, user_info);
}

boost::posix_time::ptime ItemsCacheProxy::getExpirationDate(const std::string& item_name, const UserInfo& user_info) const
{
    return cache_->getExpirationDate(item_name, user_info);
}

std::string ItemsCacheProxy::getItemValue(const std::string& item_name, const UserInfo& user_info) const
{
    return cache_->getItemValue(item_name, user_info);
}

std::vector<std::string> ItemsCacheProxy::getAllItems() const
{
    return cache_->getAllItems();
}

std::vector<std::string> ItemsCacheProxy::getAllEnabledItems(const UserInfo& user_info) const
{
    return cache_->getAllEnabledItems(user_info);
}


}  // end of shared_localization namespace
