#pragma once

#include "config.hpp"
#include "local_items.hpp"
#include "user_info.hpp"

#include <memory>

#include <boost/date_time/posix_time/posix_time.hpp>


namespace shared_localization
{

class ItemsCacheProxy;

class ItemsCache
{
public:
    static ItemsCacheProxy getCache(const std::string& project_name, const std::string& config_path, const std::string& mongo_uri, const std::string& db_name);
    static bool allAreReady();
    static void setExecutablePath(const std::string& path);

    ItemsCache(const ItemsCache& other) = delete;
    ItemsCache& operator= (const ItemsCache& other) = delete;

    ItemsCache(ItemsCache&& other) = delete;
    ItemsCache& operator= (ItemsCache&& other) = delete;

    ~ItemsCache();

    bool isReady() const;
    bool isItemEnabled(const std::string& item_name, const UserInfo& user_info) const;
    boost::posix_time::ptime getExpirationDate(const std::string& item_name, const UserInfo& user_info) const;
    std::string getItemValue(const std::string& item_name, const UserInfo& user_info) const;
    std::vector<std::string> getAllItems() const;
    std::vector<std::string> getAllEnabledItems(const UserInfo& user_info) const;
private:
    friend ItemsCacheProxy;
    using InstancesMap = std::map<std::string, std::weak_ptr<ItemsCache>>;
    static InstancesMap instances_;
    static std::string executable_path;

    ItemsCache(const std::string& project_name, const std::string& config_path, const std::string& mongo_uri, const std::string& db_name);

    Config config_;
    mutable std::uint64_t requests_after_last_manager_check = 0;

    bool tryGetSharedContainer() const;
    void checkManagerStatus() const;

    mutable std::unique_ptr<boost::interprocess::managed_shared_memory> segment_;
    mutable boost::optional<LocalItemsContainer *> container_;
};

class ItemsCacheProxy
{
public:
    ItemsCacheProxy(const ItemsCacheProxy& other);
    ItemsCacheProxy& operator= (const ItemsCacheProxy& other) = delete;

    ItemsCacheProxy(ItemsCacheProxy&& other);
    ItemsCacheProxy& operator= (ItemsCacheProxy&& other) = delete;

    bool isReady() const;
    bool isItemEnabled(const std::string& item_name, const UserInfo& user_info) const;
    boost::posix_time::ptime getExpirationDate(const std::string& item_name, const UserInfo& user_info) const;
    std::string getItemValue(const std::string& item_name, const UserInfo& user_info) const;
    std::vector<std::string> getAllItems() const;
    std::vector<std::string> getAllEnabledItems(const UserInfo& user_info) const;
private:
    friend ItemsCache;
    explicit ItemsCacheProxy(const std::shared_ptr<ItemsCache>& ptr);
    std::shared_ptr<ItemsCache> cache_;
};

}  // end of shared_localization namespace

