#include "local_items_db.hpp"
#include <boost/date_time/posix_time/ptime.hpp>

#include <iostream>
#include <bsoncxx/stdx/string_view.hpp>
#include <bsoncxx/array/element.hpp>
#include <bsoncxx/exception/exception.hpp>
#include <iterator>

#include <Poco/Logger.h>
#include <Poco/LogStream.h>

#include <mongocxx/client.hpp>
#include <bsoncxx/json.hpp>
#include <bsoncxx/builder/stream/document.hpp>


namespace shared_localization {

    template<typename T>
    bool is_null(const T &elem) {
        try {
            elem.get_null();
            return true;
        }
        catch (const bsoncxx::exception &) {
            return false;
        }
    }

    template<typename T>
    std::string value_to_string(const T &elem) {
        return elem.get_utf8().value.to_string();
    }

    void setShmStringValueOrThrow(const bsoncxx::document::element &source, ShmString &destination, const std::string& elem_name) {
        if (source && !is_null(source)) {
            if (source.type() != bsoncxx::type::k_utf8)
                throw LocalizationFormatError(elem_name + " type is not a string");

            destination = value_to_string(source).c_str();
        }
    }

    namespace {
        template <typename T>
        bsoncxx::document::element getElementOrThrow(const T &view, const char *field_name) {
            const bsoncxx::document::element elem = view[field_name];
            if (!elem)
                throw LocalizationFormatError("Cannot find BSON element '" + std::string(field_name) + "'");

            return elem;
        }

        template <typename T>
        bool ensureType(const T& element, bsoncxx::type type) {
            if (!element)
                return false;
            if (element.type() == type)
                return true;

            throw LocalizationFormatError("incorrect type");
        }

        template <typename T>
        void processField(const T& bool_el, bool &flag, UniversalAllocator &allocator) {
            if (ensureType(bool_el, bsoncxx::type::k_bool))
                flag = bool_el.get_bool();
        }

        template <typename T>
        void processField(const T& float_el, double &value, UniversalAllocator &allocator) {
            if (float_el)
                value = float_el.get_double();
        }

        template <typename T>
        void processField(const T& int_el, int &value, UniversalAllocator &allocator) {
            if (int_el)
                value = int_el.get_int32();
        }

        template <typename T>
        void processField(const T& int_el, unsigned int &value, UniversalAllocator &allocator) {
            if (int_el)
                value = (unsigned int)int_el.get_int32();
        }

        template <typename T>
        void processField(const T& string_el, ShmString &str, UniversalAllocator &allocator) {
            if (ensureType(string_el, bsoncxx::type::k_utf8))
                str = value_to_string(string_el).c_str();
        }

        template <typename T>
        void processField(const T& date_el, boost::posix_time::ptime &object, UniversalAllocator &allocator) {
            if (ensureType(date_el, bsoncxx::type::k_date))
                object = boost::posix_time::from_time_t(time_t(date_el.get_date().value.count() / 1000));
        }

        template<typename T, typename A>
        void processField(const bsoncxx::document::element &array_el, boost::interprocess::vector<T, A> &objects,
                          UniversalAllocator &allocator) {
            if (ensureType(array_el, bsoncxx::type::k_array)) {
                const auto array = array_el.get_array().value;
                objects.reserve(std::distance(array.begin(), array.end()));
                for (const auto &element: array) {
                    auto temp = T(allocator);
                    processField(element, temp, allocator);
                    objects.push_back(temp);
                }
            }
        }

        void processField(const bsoncxx::document::element &array_el, ShmUIntSet &objects,
                          UniversalAllocator &allocator) {
            if (ensureType(array_el, bsoncxx::type::k_array)) {
                for (const auto &element: array_el.get_array().value) {
                    auto temp = ((UIntAllocator&)allocator).allocate(1);
                    processField(element, *temp, allocator);
                    objects.insert(*temp);
                }
            }
        }

        template<typename T>
        void processSubfield(const bsoncxx::document::element &dict_el, const std::string &subfield_name, T &object,
                             UniversalAllocator &allocator) {
            if (ensureType(dict_el, bsoncxx::type::k_document)) {
                processField(dict_el[subfield_name], object, allocator);
            }
        }

        template <typename Element>
        void processApplication(
                const Element& application_el,
                ItemLocalization::EnableConditions::Application &application,
                UniversalAllocator &allocator,
                const ApplicationsCache &apps_cache) {
            if (ensureType(application_el, bsoncxx::type::k_document)) {
                const std::string application_name = value_to_string(getElementOrThrow(application_el, "name"));
                application.aliases.emplace(application_name.c_str(), allocator);
                auto it = apps_cache.find(application_name);
                if (it != apps_cache.cend())
                    for (const auto &alias: it->second)
                        application.aliases.emplace(alias.c_str(), allocator);

                processSubfield(application_el["version"], "from", application.version.from, allocator);
                processSubfield(application_el["version"], "to", application.version.to, allocator);
            }
        }

        void processApplications(
                const bsoncxx::document::element &applications_el,
                ItemLocalization::EnableConditions::Applications &applications,
                UniversalAllocator &allocator,
                const ApplicationsCache &apps_cache) {
            if (ensureType(applications_el, bsoncxx::type::k_array)) {
                auto elements = applications_el.get_array().value;

                applications.resize(
                        std::distance(elements.begin(), elements.end()),
                        ItemLocalization::EnableConditions::Application(allocator)
                );

                size_t i = 0;
                for (auto elem: elements)
                    processApplication(elem, applications[i++], allocator, apps_cache);
            }
        }

        void processConditions(
                const bsoncxx::document::element &conditions_el,
                ItemLocalization::EnableConditions &conditions,
                UniversalAllocator &allocator,
                const ApplicationsCache &apps_cache) {
            // Enabled
            conditions.enabled = true;
            processField(conditions_el["enabled"], conditions.enabled, allocator);
            // Device types
            processField(conditions_el["deviceTypes"], conditions.device_types, allocator);
            // Uuids
            processField(conditions_el["uuids"], conditions.uuids, allocator);
            // Language
            processSubfield(conditions_el["locale"], "language", conditions.language, allocator);
            if (conditions.language.empty())
                conditions.language = "*";
            boost::to_lower(conditions.language);
            // Country
            processSubfield(conditions_el["locale"], "country", conditions.country, allocator);
            if (conditions.country.empty())
                conditions.country = "*";
            boost::to_lower(conditions.country);
            // Time
            conditions.time_start = boost::posix_time::neg_infin;
            processSubfield(conditions_el["time"], "from", conditions.time_start, allocator);
            conditions.time_end = boost::posix_time::pos_infin;
            processSubfield(conditions_el["time"], "to", conditions.time_end, allocator);
            // Applications
            processApplications(conditions_el["applications"], conditions.applications, allocator, apps_cache);
            // Init region ids
            processField(conditions_el["region_ids_init"], conditions.region_ids_init, allocator);
            // Init region ids blacklist
            processField(conditions_el["region_ids_init_blacklist"], conditions.region_ids_init_blacklist, allocator);
            // Actual region ids
            processField(conditions_el["region_ids"], conditions.region_ids, allocator);
            // Actual region ids blacklist
            processField(conditions_el["region_ids_blacklist"], conditions.region_ids_blacklist, allocator);

            // Audience
            conditions.audience_ratio = 1.0;
            processField(conditions_el["audience"], conditions.audience_ratio, allocator);
            conditions.audience_offset = 0.0;
            processField(conditions_el["audience_offset"], conditions.audience_offset, allocator);

            // Clids
            const bsoncxx::document::element clids_el = conditions_el["clids"];
            if (clids_el) {
                for (const auto &clid_el: clids_el.get_array().value) {
                    std::list<unsigned int> values;
                    if (clid_el["value"].type() == bsoncxx::type::k_array) {
                        for (const auto &clid_value: clid_el["value"].get_array().value)
                            values.push_back(clid_value.get_int32());
                    } else {
                        if (clid_el["value"].type() != bsoncxx::type::k_int32)
                            throw std::runtime_error("Expected clid's value of type: array or integer");
                        values.push_back(clid_el["value"].get_int32());
                    }
                    unsigned int clid_number = clid_el["number"].get_int32();
                    ShmUIntSet clid_values(values.begin(), values.end(), ShmUIntSet::key_compare(), allocator);
                    conditions.clids.insert(std::make_pair(clid_number, clid_values));
                }
            }
            // Models
            const auto models_el = conditions_el["models"];
            if (models_el) {
                const auto models = models_el.get_array().value;
                conditions.models.resize(
                        std::distance(models.begin(), models.end()),
                        ItemLocalization::EnableConditions::Model(allocator));

                size_t i = 0;
                for (const auto& model: models) {
                    processField(model["vendor"], conditions.models[i].vendor, allocator);
                    processField(model["model"], conditions.models[i].name, allocator);
                    boost::to_lower(conditions.models[i].vendor);
                    boost::to_lower(conditions.models[i].name);
                    ++i;
                }
            }

            // Extended Params
            const auto extended_params_el = conditions_el["extendedParams"];
            if (extended_params_el && extended_params_el.type() == bsoncxx::type::k_document)
                for (const auto& el: extended_params_el.get_document().value)
                    if (el.type() == bsoncxx::type::k_utf8) {
                        ShmString name(el.key().to_string().c_str(), allocator);
                        ShmString value(value_to_string(el).c_str(), allocator);
                        conditions.extended_params.insert(std::make_pair(name, value));
                    }
        }

        void processValue(
                const bsoncxx::array::element &v_el,
                ItemLocalization &item_localization,
                UniversalAllocator &allocator,
                const ApplicationsCache &apps_cache) {

            ShmString value(allocator);
            setShmStringValueOrThrow(v_el["value"], value, "value");
            item_localization.value = value;


            const bsoncxx::document::element conditions_el = v_el["conditions"];
            if (conditions_el && !is_null(conditions_el))
                processConditions(conditions_el, item_localization.conditions, allocator, apps_cache);
        }

        void processOptions(const bsoncxx::document::element &options_el, ItemOptions &item_options, const UniversalAllocator& allocator) {
            if (!options_el)
                return;

            ShmString audience_salt(allocator);
            setShmStringValueOrThrow(options_el["audience_salt"], audience_salt, "audience_salt");
            if (!audience_salt.empty())
                item_options.audience_salt = audience_salt;
        }
    }  // end of anonymous namespace

    void processItem(const bsoncxx::document::view &view, SharedLocalItems &shared_items,
                     const ApplicationsCache &apps_cache) {
        ShmString item_id(shared_items.allocator);
        std::string id = value_to_string(getElementOrThrow(view, "_id"));
        item_id = id.c_str();
        const bsoncxx::document::element values_el = view["values"];
        if (!values_el)
            return;

        auto item_localizations = ItemLocalizations(shared_items.allocator);
        try {
            for (auto v_el: values_el.get_array().value) {
                ItemLocalization item_localization(shared_items.allocator);
                processValue(v_el, item_localization, shared_items.allocator, apps_cache);
                item_localizations.push_back(item_localization);
            }
        }
        catch (const std::exception &e) {
            throw std::runtime_error("Unable to process item " + id + ": " + e.what());
        }

        const bsoncxx::document::element options_el = view["options"];
        auto item_options = ItemOptions{};
        processOptions(options_el, item_options, shared_items.allocator);

        shared_items.items.insert(std::make_pair(item_id, std::make_pair(item_options, item_localizations)));
    }

    void processYandexApp(const bsoncxx::document::view &view, ApplicationsCache &cache) {
        const std::string application_name = value_to_string(getElementOrThrow(view, "_id"));
        auto& app = cache[application_name];

        if (view.find("ios_bundle") != view.end())
            app.insert(value_to_string(view["ios_bundle"]));

        if (view.find("android_package_name") != view.end())
            app.insert(value_to_string(view["android_package_name"]));

        if (view.find("wp_app_id") != view.end())
            app.insert(value_to_string(view["wp_app_id"]));
    }

    LocalItemsDb::LocalItemsDb(const Config &cfg)
            : cfg_(cfg) {}

    void LocalItemsDb::dump(const std::string &project_name, SharedLocalItems &items_cache) {
        Poco::LogStream log_stream(Poco::Logger::get("LocalItemsDB"));
        mongocxx::client client{mongocxx::uri{cfg_.mongo_uri}};
        doDump(client[cfg_.db_name], project_name, items_cache);
    }

    void LocalItemsDb::doDump(
            mongocxx::database db,
            const std::string &project_name,
            SharedLocalItems &items_cache) {
        using bsoncxx::builder::stream::document;
        using bsoncxx::builder::stream::finalize;

        Poco::LogStream log_stream(Poco::Logger::get("LocalItemsDB"));
        ApplicationsCache apps_cache;
        // Populate apps cache
        {
            log_stream.debug() << "Building apps cache" << std::endl;

            mongocxx::cursor cursor = db[cfg_.getCollectionName(cfg_.applications_collection_subname)].find({});
            for (auto doc_view : cursor) {
                processYandexApp(doc_view, apps_cache);
            }

            log_stream.debug() << "Apps cache built" << std::endl;
        }

        // Items themselves
        {
            std::string collection_name = cfg_.getCollectionName(project_name);
            log_stream.debug() << "Dumping collection " << collection_name << std::endl;
            mongocxx::cursor cursor = db[collection_name].find({});

            for (auto doc: cursor) {
                processItem(doc, items_cache, apps_cache);
            }

            log_stream.debug() << "Project " << project_name << " dumped" << std::endl;
        }
    }

}  // end of shared_localization namespace

