#include "config.hpp"
#include "daemon.hpp"
#include "local_items.hpp"
#include "local_items_db.hpp"

#include <csignal>
#include <cstring>

#include <boost/interprocess/sync/sharable_lock.hpp>

#include <Poco/AutoPtr.h>
#include <Poco/FormattingChannel.h>
#include <Poco/Logger.h>
#include <Poco/LogStream.h>
#include <Poco/PatternFormatter.h>
#include <Poco/SimpleFileChannel.h>

#include <mongocxx/client.hpp>
#include <mongocxx/instance.hpp>
#include <mongocxx/exception/exception.hpp>

using namespace shared_localization;

inline bool process_is_alive(pid_t pid) {
    return ((kill(pid, 0) == 0) || (errno != ESRCH));
}

volatile bool shutdown_requested = false;

void handle_terminate(int /* code */, siginfo_t *siginfo, void *) {
    pid_t sender = siginfo->si_pid;
    Poco::LogStream log_stream(Poco::Logger::get("SLManager"));
    log_stream.debug() << "SIGTERM received from " << sender << std::endl;
    shutdown_requested = true;
}

void handle_interrupt(int /* code */, siginfo_t *siginfo, void *) {
    pid_t sender = siginfo->si_pid;
    Poco::LogStream log_stream(Poco::Logger::get("SLManager"));
    log_stream.debug() << "SIGINT received from " << sender << std::endl;
    shutdown_requested = true;
}

void handle_client_shutdown(int /* code */, siginfo_t *siginfo, void *) {
    pid_t sender = siginfo->si_pid;
    Poco::LogStream log_stream(Poco::Logger::get("SLManager"));
    log_stream.debug() << "SIGUSR1 received from " << sender << std::endl;
    // Do nothing (except for interrupting sleep call)
}

void handle_reopen_logs(int /* code */, siginfo_t *siginfo, void *) {
    pid_t sender = siginfo->si_pid;
    auto &logger = Poco::Logger::get("SLManager");
    Poco::LogStream log_stream(logger);
    log_stream.debug() << "SIGUSR2 received from " << sender << std::endl;
    logger.getChannel()->close();
    logger.getChannel()->open();
    log_stream.information() << "Log reopened" << std::endl;
}

void actualize_clients(shared_localization::ShmIntSet &clients,
                       boost::interprocess::interprocess_sharable_mutex &clients_mutex,
                       Poco::LogStream &log_stream);

void init_signals(struct sigaction &sig_term) {
    sig_term.sa_sigaction = handle_terminate;
    sig_term.sa_flags = SA_SIGINFO;
    sigaction(SIGTERM, &sig_term, NULL);
    struct sigaction sig_int;
    sig_int.sa_sigaction = handle_interrupt;
    sig_int.sa_flags = SA_SIGINFO;
    sigaction(SIGINT, &sig_int, NULL);
    struct sigaction sig_usr1;
    sig_usr1.sa_sigaction = handle_client_shutdown;
    sig_usr1.sa_flags = SA_SIGINFO;
    sigaction(SIGUSR1, &sig_usr1, NULL);
    struct sigaction sig_usr2;
    sig_usr2.sa_sigaction = handle_reopen_logs;
    sig_usr2.sa_flags = SA_SIGINFO;
    sigaction(SIGUSR2, &sig_usr2, NULL);
}

// Cleanup
struct Cleanup {
    explicit Cleanup(const Config &cfg)
            : cfg_(cfg) {}

    ~Cleanup() {
        pidfile_cleanup(cfg_.getPidfilePath());  // delete pidfile
        boost::interprocess::shared_memory_object::remove(cfg_.memory_segment_name.c_str());
    }

private:
    const Config &cfg_;
};


int main(int argc, char *argv[]) {
    // Version
    if (argc == 2 && std::strcmp(argv[1], "-V") == 0)
    {
        std::cout << argv[0] << " version: " << SHARED_LOCALIZATION_VERSION << std::endl;
        return EXIT_FAILURE;
    }

    // Usage
    if (argc != 5) {
        std::cerr << "Usage: " << argv[0] << " {PROJECT} {CONFIG} {MONGO_URI} {DB_NAME}" << std::endl;
        std::cerr << "-V: print the version number and exit" << std::endl;
        return EXIT_FAILURE;
    }

    // Set umask
    umask(0033);  // all for user, only read for group/other

    // Load config
    Config config;
    config.project_name = argv[1];
    config.loadFromFile(argv[2]);
    config.mongo_uri = argv[3];
    config.db_name = argv[4];
    config.validate();

    // Initialize logger
    Poco::AutoPtr<Poco::SimpleFileChannel> channel(new Poco::SimpleFileChannel);
    channel->setProperty("path", config.logfile_path);
    channel->open();
    Poco::AutoPtr<Poco::PatternFormatter> formatter(new Poco::PatternFormatter);
    formatter->setProperty("pattern", config.log_format);
    Poco::AutoPtr<Poco::FormattingChannel> formatting_channel(
            new Poco::FormattingChannel(formatter, channel));
    Poco::Logger::root().setChannel(formatting_channel);
    Poco::Logger::root().setLevel(config.log_level);

    auto &logger = Poco::Logger::get("SLManager");
    Poco::LogStream log_stream(logger);

    // Set pid file
    set_pid_file(config.getLockfilePath(), config.getPidfilePath());

    // We need new process group, so we won't get killed by init restarting service
    setpgid(0, 0);

    // Signal control
    struct sigaction sig_term;
    init_signals(sig_term);

    // Start managing
    log_stream.information() << "New manager initialized. Version: " << SHARED_LOCALIZATION_VERSION
                             << " Pid: " << getpid() << std::endl;
    try {
        Cleanup cleanup(config);
        // Create named shared memory segment
        boost::interprocess::managed_shared_memory segment;

        mongocxx::instance instance{};

        try {
            segment = boost::interprocess::managed_shared_memory(
                    boost::interprocess::create_only,
                    config.memory_segment_name.c_str(),
                    config.memory_limit);
        }
        catch (const boost::interprocess::interprocess_exception &e) {
            try {
                boost::interprocess::shared_memory_object::remove(config.memory_segment_name.c_str());
                segment = boost::interprocess::managed_shared_memory(
                        boost::interprocess::create_only,
                        config.memory_segment_name.c_str(),
                        config.memory_limit);
            }
            catch (const boost::interprocess::interprocess_exception &) {
                log_stream.fatal() << "Unable to acquire memory segment: " << e.what() << std::endl;
                throw e;
            }
            log_stream.warning() << "Shared memory already existed, purged and recreated" << std::endl;
        }
        log_stream.information() << "Shared memory acquired successfully" << std::endl;

        // Construct shared container in memory segment
        LocalItemsContainer *shared_container;
        try {
            shared_container = segment.construct<LocalItemsContainer>("container")(
                    segment,
                    config.bucket_number,
                    SHARED_LOCALIZATION_VERSION);
        }
        catch (const boost::interprocess::interprocess_exception &e) {
            log_stream.fatal() << "Unable to construct shared container: " << e.what() << std::endl;
            throw;
        }
        log_stream.information() << "Data shared container constructed successfully" << std::endl;

        // Add parents pid to the clients set
        {
            boost::interprocess::scoped_lock<boost::interprocess::interprocess_sharable_mutex> lock(
                    shared_container->clients_mutex);
            auto ppid = getppid();
            if (ppid != 0)  // somewhat undocumented thing: means parent is already dead
            {
                log_stream.debug() << "Added " << ppid << "(parent) to clients" << std::endl;
                shared_container->clients.insert(ppid);
            } else {
                log_stream.warning() << "getppid() returned 0. Is parent dead?" << std::endl;
            }
        }

        // Initialize container with content
        auto db = LocalItemsDb(config);
        SharedLocalItems shared_items(segment);
        try {
            while (!shared_container->is_ready && !shutdown_requested) {
                log_stream.information() << "Performing initial population of cache" << std::endl;
                try {
                    shared_items.items.clear();
                    db.dump(config.project_name, shared_items);
                }
                catch (const mongocxx::exception &e) {
                    log_stream.error() << "Unable to populate cache due to mongo error: " << e.what() << std::endl;
                    usleep(config.mongo_retry_timeout);
                    continue;
                }
                log_stream.debug() << "Dumped " << config.project_name << " collection from mongo" << std::endl;
                shared_container->cache.swap(shared_items.items);
                log_stream.information() << "Cache is populated with " << shared_container->cache.size() << " items"
                                         << std::endl;
                shared_container->is_ready = true;
                log_stream.debug() << "Shared items cleaned up. Cache is ready" << std::endl;
            }
        }
        catch (const boost::interprocess::bad_alloc &) {
            log_stream.fatal() << "Unable to populate cache due to lack of memory! Exiting" << std::endl;
            throw;
        }
        catch (const std::exception &e) {
            log_stream.fatal() << "Unable to populate cache due to exception: " << e.what() << ". Exiting" << std::endl;
            throw;
        }
        catch (...) {
            log_stream.fatal() << "Unable to populate cache due to unknown error. Exiting." << std::endl;
            throw;
        }

        // Start updating cycle
        while (!shutdown_requested) {
            actualize_clients(shared_container->clients, shared_container->clients_mutex, log_stream);
            if (shared_container->clients.empty() || shutdown_requested)
                break;

            auto left_to_sleep = sleep(config.update_interval);
            while (left_to_sleep) {
                actualize_clients(shared_container->clients, shared_container->clients_mutex, log_stream);
                if (shared_container->clients.empty() || shutdown_requested)
                    break;
                left_to_sleep = sleep(left_to_sleep);
            }
            if (left_to_sleep)
                break;

            // Check is there are any clients
            actualize_clients(shared_container->clients, shared_container->clients_mutex, log_stream);
            if (shared_container->clients.empty() || shutdown_requested)
                break;
            log_stream.debug() << "Proceed execution, still " << shared_container->clients.size() << " clients left"
                               << std::endl;

            // Perform update
            log_stream.information() << "Starting update" << std::endl;
            try {
                shared_items.items.clear();
                db.dump(config.project_name, shared_items);
                log_stream.debug() << "Dumped " << config.project_name << " from mongo" << std::endl;
                {
                    boost::interprocess::scoped_lock<boost::interprocess::interprocess_sharable_mutex> lock(
                            shared_container->cache_mutex);
                    shared_container->cache.swap(shared_items.items);
                }
                log_stream.information() << "Updated cache. Found " << shared_container->cache.size() << " items"
                                         << std::endl;
            }
            catch (const mongocxx::exception &e) {
                log_stream.error() << "Unable to update cache due to mongo error: " << e.what() << std::endl;
            }
            catch (const boost::interprocess::bad_alloc &) {
                log_stream.error() << "Unable to update cache due to lack of memory!" << std::endl;
            }
            catch (const std::exception &e) {
                log_stream.error() << "Unable to update cache due to exception: " << e.what()
                                   << ". Using already existing cache" << std::endl;
            }
            catch (...) {
                log_stream.error() << "Unable to update cache due to unknown error. Using already existing cache"
                                   << std::endl;
            }
        }
    }
    catch (const std::exception &e) {
        log_stream.fatal() << "Unhandled exception during shm manager execution: " << e.what() << std::endl;
        return EXIT_FAILURE;
    }
    catch (...) {
        log_stream.fatal() << "Unknown error during shm manager execution" << std::endl;
        return EXIT_FAILURE;
    }
    if (shutdown_requested)
        log_stream.information() << "Shutdown requested, shutting down" << std::endl;
    else
        log_stream.information() << "No more clients, shutting down" << std::endl;
    return EXIT_SUCCESS;
}

void actualize_clients(shared_localization::ShmIntSet &clients,
                       boost::interprocess::interprocess_sharable_mutex &clients_mutex,
                       Poco::LogStream &log_stream) {
    for (auto client_pid_it = clients.begin(); client_pid_it != clients.end();) {
        if (process_is_alive(*client_pid_it)) {
            ++client_pid_it;
        } else {
            log_stream.warning() << *client_pid_it << " is dead, removing from clients" << std::endl;
            boost::interprocess::scoped_lock<boost::interprocess::interprocess_sharable_mutex> lock(clients_mutex);
            client_pid_it = clients.erase(client_pid_it);
        }
    }
}
