#include "user_info.hpp"

#include <boost/algorithm/string/case_conv.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/join.hpp>

#include <boost/python/stl_iterator.hpp>
namespace py = boost::python;

#include <algorithm>
#include <string>
#include <gmpxx.h>
#include <openssl/md5.h>

namespace shared_localization {

    template<typename Container>
    inline bool contains(const typename Container::value_type& value, const Container& container)
    {
        return container.find(value) != container.end();
    }


    template<typename T>
    inline std::vector<T> to_std_vector(const py::list& iterable)
    {
        return std::vector<T>(py::stl_input_iterator<T>(iterable), py::stl_input_iterator<T>());
    }

    std::string computeMD5(const std::string &data) {
        unsigned char checksumBuffer[MD5_DIGEST_LENGTH];
        MD5((const unsigned char *) data.c_str(), data.length(), checksumBuffer);
        std::stringstream result;
        result << std::hex << std::setfill('0');
        for (unsigned int v: checksumBuffer)
            result << std::setw(2) << v;
        return result.str();
    }

    inline std::string shm_to_string(const ShmString &shm_string) {
        return std::string(shm_string.data(), shm_string.size());
    }

    bool is_number(const std::string &value) {
        try {
            std::stoul(value);
            return true;
        }
        catch (const std::invalid_argument &) {
        }
        catch (const std::out_of_range &) {
        }

        return false;
    }

    bool both_have_equal_type(bool a, bool b) {
        return a == b;
    }

    void splitVersion(const std::string &str_version, std::vector<std::string> &version) {
        version.clear();
        if (!str_version.empty())
            boost::split(version, str_version, boost::is_any_of(".-"));
    }

    void splitVersion(const ShmString &str_version, std::vector<std::string> &version) {
        std::string v = shm_to_string(str_version);
        splitVersion(v, version);
    }

    bool checkConditions(
            const std::string &from,
            const std::string &to,
            const boost::optional<std::string> &user_version) {
        if (from.empty() && to.empty())
            return true;

        if (!user_version.is_initialized())
            return false;

        std::vector<std::string> v_version, v_lower, v_upper;
        splitVersion(*user_version, v_version);
        splitVersion(from, v_lower);
        splitVersion(to, v_upper);

        while (v_version.size() < v_lower.size() || v_version.size() < v_upper.size())
            v_version.push_back("0");

        for (size_t i = 0; i < v_version.size() && i < v_lower.size(); ++i) {
            bool lower_bound_is_number = is_number(v_lower[i]);
            bool version_is_number = is_number(v_version[i]);

            if (both_have_equal_type(lower_bound_is_number, version_is_number)) {
                if (v_lower[i] < v_version[i])
                    break;
                if (v_lower[i] > v_version[i])
                    return false;
            } else if (lower_bound_is_number)
                return false;
        }

        for (size_t i = 0; i < v_version.size() && i < v_upper.size(); ++i) {
            bool upper_bound_is_number = is_number(v_upper[i]);
            bool version_is_number = is_number(v_version[i]);

            if (both_have_equal_type(upper_bound_is_number, version_is_number)) {
                if (v_version[i] < v_upper[i])
                    break;
                if (v_version[i] > v_upper[i])
                    return false;
            } else if (!upper_bound_is_number)
                return false;
        }

        return true;
    }

    namespace {

        template <typename T>
        std::ostream& operator<< (std::ostream& stream, const boost::optional<T>& value) {
            if (value.is_initialized())
                return stream << *value;
            return stream << "<empty>";
        }

        std::ostream& operator<< (std::ostream& stream, const boost::optional<std::vector<int> >& value) {
            if (value.is_initialized()) {
                stream << "[";
                for (auto v: *value)
                    stream << v << ", ";
                return stream << "]";
            }
            else {
                return stream << "<empty>";
            }
        }

        std::ostream& operator<< (std::ostream& stream, const ShmUIntSet& value) {
            stream << "set([";
            for (auto v: value)
                stream << v << ", ";
            return stream << "])";
        }

        std::ostream& operator<< (std::ostream& stream, const std::map<unsigned long, unsigned long>& value) {
            stream << "{";
            for (auto pair: value)
                stream << pair.first << ": " << pair.second  << ", ";
            return stream << "}";
        }

        bool timeMatchesCondition(
                const boost::posix_time::ptime &time_start,
                const boost::posix_time::ptime &time_end) {
            const boost::posix_time::ptime current_time(boost::posix_time::microsec_clock::universal_time());
            return time_start <= current_time && current_time < time_end;
        }

        inline void validateUUID(const std::string &uuid) {
            if (uuid.length() != 32) throw std::runtime_error("UUID is incorrect");
            for (size_t i = 0; i < uuid.length(); ++i) {
                if (!isxdigit(uuid[i])) throw std::runtime_error("UUID is incorrect");
            }
        }

        bool localePartMatchesConditionPart(
                const boost::optional<std::string> &locale_part,
                const ShmString &condition_part) {
            try {
                if (condition_part == "*") {
                    return true;
                } else if (locale_part.is_initialized()) {
                    if (condition_part.at(0) == '!')
                        return condition_part.substr(1, 2) != locale_part.get().c_str();
                    else
                        return condition_part.substr(0, 2) == locale_part.get().c_str();
                }
            }
            catch (...) {}
            return false;
        }

        const mpq_class max_uuid_value(std::string(32, 'F'), 16);

        double calcUuidAudienceRatio(const std::string &uuid, const std::string &audience_salt) {
            const mpq_class ratio = mpq_class(computeMD5(audience_salt + uuid), 16) / max_uuid_value;
            return ratio.get_d();
        }

        double calcUuidAudienceRatio(const std::string &uuid) {
            const mpq_class ratio = mpq_class(uuid, 16) / max_uuid_value;
            return ratio.get_d();
        }

        bool versionMatchesCondition(
                const ItemLocalization::EnableConditions::VersionRange &version_range,
                const boost::optional<std::string> &version) {
            return checkConditions(shm_to_string(version_range.from), shm_to_string(version_range.to),
                                   version);
        }

    }// end of anonymous namespace

    UserInfo::UserInfo() {}

    UserInfo &UserInfo::setLocale(const std::string &locale) {
        if (locale.length() >= 2) {
            language_ = locale.substr(0, 2);
            boost::to_lower(language_.get());
            if (locale.length() >= 5) {
                country_ = locale.substr(3, 2);
                boost::to_lower(country_.get());
            } else {
                country_.reset();
            }
        } else {
            language_.reset();
            country_.reset();
        }
        return *this;
    }

    UserInfo &UserInfo::setLocale(const std::string &language, const std::string &country) {
        if (language.empty()) {
            language_.reset();
        } else {
            language_ = language;
            boost::to_lower(language_.get());
        }

        if (country.empty()) {
            country_.reset();
        } else {
            country_ = country;
            boost::to_lower(country_.get());
        }
        return *this;
    }

    UserInfo &UserInfo::setRegionIds(const py::list& region_ids) {
        region_ids_ = to_std_vector<int>(region_ids);
        return *this;
    }

    UserInfo &UserInfo::setRegionIdsInit(const py::list& region_ids_init) {
        region_ids_init_ = to_std_vector<int>(region_ids_init);
        return *this;
    }

    UserInfo &UserInfo::setClid(unsigned long clidNumber, unsigned long clidValue) {
        clids_[clidNumber] = clidValue;
        return *this;
    }

    UserInfo &UserInfo::setUuid(const std::string &uuid) {
        validateUUID(uuid);
        uuid_ = boost::to_lower_copy(uuid);
        uuid_audience_ratio_.reset();
        uuid_salted_audience_ratio_.reset();
        return *this;
    }

    UserInfo &UserInfo::setDeviceType(const std::string &device_type) {
        device_type_ = device_type;
        return *this;
    }

    UserInfo &UserInfo::setModel(const std::string &vendor, const std::string &name) {
        vendor_ = boost::to_lower_copy(vendor);
        model_ = boost::to_lower_copy(name);
        return *this;
    }

    UserInfo &UserInfo::setApplication(const std::string &name, const std::string &version) {
        application_name_ = name;
        application_version_ = version;
        return *this;
    }

    UserInfo &UserInfo::setScreenSize(int width, int height) {
        screen_width_ = width;
        screen_height_ = height;
        return *this;
    }

    UserInfo &UserInfo::setScreenDpi(int dpi) {
        screen_dpi_ = dpi;
        return *this;
    }

    UserInfo &UserInfo::setExtendedParam(
            const std::string &name,
            const std::string &value,
            UserInfo::ExtendedParamsComparatorType comparator) {
        if (!comparator)
            throw std::runtime_error("Got NULL as setExtendedParam comparator function");
        MatchExtParam &ep = match_ext_params_[name];

        ep.value = value;
        ep.comparator = comparator;

        return *this;
    }


    bool UserInfo::matchesConditions(const ItemLocalization::EnableConditions &conditions,
                                     const ItemOptions &options) const {
        if (!conditions.enabled)
            return false;

        return localeMatchesCondition(conditions.language, conditions.country) &&

               regionIdsInitMatchesCondition(conditions.region_ids_init, conditions.region_ids_init_blacklist) &&

               regionIdsMatchesCondition(conditions.region_ids, conditions.region_ids_blacklist) &&

               applicationMatchesCondition(conditions.applications) &&

               uuidMatchesCondition(conditions.uuids) &&

               timeMatchesCondition(conditions.time_start, conditions.time_end) &&

               audienceRatioMatchesCondition(conditions.audience_ratio, conditions.audience_offset,
                                             options.audience_salt) &&

               deviceTypeMatchesCondition(conditions.device_types) &&

               modelMatchesCondition(conditions.models) &&

               clidsMatchesCondition(conditions.clids) &&

               extendedParamsMatchesCondition(conditions.extended_params);
    }

    bool UserInfo::uuidMatchesCondition(const ShmStringVector &uuids) const {
        if (uuids.empty())
            return true;

        if (!uuid_)
            return false;

        for (size_t i = 0; i < uuids.size(); ++i) {
            if (uuid_.get().c_str() == boost::to_lower_copy(shm_to_string(uuids[i])))
                return true;
        }

        return false;
    };

    bool UserInfo::localeMatchesCondition(
            const ShmString &conditional_language,
            const ShmString &conditional_country) const {
        return localePartMatchesConditionPart(language_, conditional_language) &&
               localePartMatchesConditionPart(country_, conditional_country);
    }

    bool UserInfo::regionIdsMatchesCondition(const ShmUIntSet &localization_region_ids,
                                             const ShmUIntSet &localization_region_ids_blacklist) const {
        return isGeoTargetingSuccessed(
                region_ids_,
                localization_region_ids,
                localization_region_ids_blacklist);
    }

    bool UserInfo::regionIdsInitMatchesCondition(const ShmUIntSet &localization_region_ids_init,
                                                 const ShmUIntSet &localization_region_ids_init_blacklist) const {
        return isGeoTargetingSuccessed(
                region_ids_init_,
                localization_region_ids_init,
                localization_region_ids_init_blacklist);
    }

    bool UserInfo::isGeoTargetingSuccessed(
            const boost::optional <std::vector<int>> &region_ids,
            const ShmUIntSet &localization_region_ids,
            const ShmUIntSet &localization_region_ids_blacklist) {

        if (localization_region_ids.empty() && localization_region_ids_blacklist.empty())
            // prevent checking uninitialized user's region ids
            return true;

        if (!region_ids.is_initialized())
            // user's region is unknown
            return false;

        auto &user_region_ids = region_ids.get();

        if (localization_region_ids.empty()) {
            // test if one of user's region_id is blacklisted
            for (const auto &user_region_id: user_region_ids) {
                if (contains(user_region_id, localization_region_ids_blacklist)) {
                    // region_id is blacklisted
                    return false;
                }
            }
            return true;
        } else {
            for (const auto &localization_region_id: localization_region_ids) {
                auto it_end = std::end(user_region_ids);
                if (std::find(std::begin(user_region_ids), it_end, localization_region_id) != it_end) {
                    if (contains(localization_region_id, localization_region_ids_blacklist)) {
                        // localization_region_id is blacklisted
                        continue;
                    }
                    return true;
                }
            }
        }
        return false;
    }

    bool UserInfo::audienceRatioMatchesCondition(
            double audience_ratio,
            double audience_offset,
            const boost::optional<ShmString> &audience_salt) const {
        const double epsilon = 0.0001;

        if (audience_ratio + epsilon >= 1.0 && audience_offset <= epsilon)
            return true;

        if (!uuid_)
            return false;

        double effective_ratio;
        if (audience_salt) {
            const std::string salt = shm_to_string(audience_salt.get());
            if ((!uuid_salted_audience_ratio_) || (salt != last_used_salt_)) {
                last_used_salt_ = salt;
                uuid_salted_audience_ratio_ = calcUuidAudienceRatio(uuid_.get(), salt);
            }
            effective_ratio = uuid_salted_audience_ratio_.get();
        } else {
            if (!uuid_audience_ratio_) {
                uuid_audience_ratio_ = calcUuidAudienceRatio(uuid_.get());
            }
            effective_ratio = uuid_audience_ratio_.get();
        }

        return ((effective_ratio >= audience_offset) && (effective_ratio <= audience_ratio + audience_offset));
    }

    bool UserInfo::deviceTypeMatchesCondition(const ShmStringVector &conditional_device_types) const {
        if (conditional_device_types.empty())
            return true;

        if (!device_type_)
            return false;

        const auto &device_type = device_type_.get().c_str();

        for (const auto &conditional_device_type: conditional_device_types) {
            if (device_type == conditional_device_type)
                return true;
        }

        return false;
    }

    bool UserInfo::modelMatchesCondition(
            const ItemLocalization::EnableConditions::Models &conditional_models) const {
        if (conditional_models.empty())
            return true;

        for (const auto &conditional_model: conditional_models) {
            if ((conditional_model.vendor.empty() || (vendor_ == shm_to_string(conditional_model.vendor)))
                && ((conditional_model.name.empty()) || (model_ == shm_to_string(conditional_model.name))))
                return true;
        }
        return false;
    }

    bool UserInfo::applicationMatchesCondition(
            const ItemLocalization::EnableConditions::Applications &applications) const {
        if (applications.empty())
            return true;

        if (!application_name_.is_initialized())
            return false;

        for (const auto &application: applications) {
            ShmString app_name(applications.get_stored_allocator());
            app_name = application_name_.get().c_str();
            if (application.aliases.count(app_name) &&
                versionMatchesCondition(application.version, application_version_))
                return true;
        }

        return false;
    }

    bool UserInfo::clidsMatchesCondition(const ItemLocalization::EnableConditions::Clids &clids) const {
        if (clids.empty())
            return true;

        for (auto conditional_clid: clids) {
            auto it = clids_.find(conditional_clid.first);
            if (it == clids_.end())
                return false;

            auto clid_values = conditional_clid.second;
            if (clid_values.find(it->second) == clid_values.cend())
                return false;
        }
        return true;
    }

    bool UserInfo::extendedParamsMatchesCondition(
            const ItemLocalization::EnableConditions::ExtendedParams &conditional_ext_params) const {
        if (conditional_ext_params.empty())
            return true;

        for (const auto &condition_param: conditional_ext_params) {
            std::string conditional_param_name = shm_to_string(condition_param.first);
            auto it = match_ext_params_.find(conditional_param_name);

            if (it == match_ext_params_.end())
                return false;

            std::string conditional_param_value = shm_to_string(condition_param.second);
            const auto &user_param = it->second;
            if (!user_param.comparator(user_param.value, conditional_param_value))
                return false;
        }
        return true;
    }

    std::string UserInfo::toString() const {
        std::stringstream stream;
        stream << "uuid=" << uuid_ << ", "
               << "language=" << language_ << ", "
               << "country=" << country_ << ", "
               << "region_ids=" << region_ids_ << ", "
               << "region_ids_init=" << region_ids_init_ << ", "
               << "deviceType=" << device_type_ << ", "
               << "clids=" << clids_ << ", "
               << "vendor=" << vendor_ << ", "
               << "model=" << model_ << ", "
               << "applicationName=" << application_name_ << ", "
               << "applicationVersion=" << application_version_ << ", "
               << "screenWidth=" << screen_width_ << ", "
               << "screenHeight=" << screen_height_ << ", "
               << "screenDpi=" << screen_dpi_;
        return stream.str();
    }

}  // end of shared_localization namespace
