#pragma once

#include "local_items.hpp"

#include <string>

#include <boost/python.hpp>
#include <boost/function.hpp>
#include <boost/optional.hpp>
#include <boost/python/stl_iterator.hpp>


namespace py = boost::python;

namespace shared_localization {
    std::string computeMD5(const std::string &data);

    bool checkConditions(
            const std::string &from,
            const std::string &to,
            const boost::optional<std::string> &user_version);

    class UserInfo {
    public:
        UserInfo();

        UserInfo &setLocale(const std::string &locale);

        UserInfo &setLocale(const std::string &language, const std::string &country);

        UserInfo &setRegionIds(const py::list& region_ids);

        UserInfo &setRegionIdsInit(const py::list& region_ids_init);

        UserInfo &setClid(unsigned long clidNumber, unsigned long clidValue);

        UserInfo &setUuid(const std::string &uuid);

        UserInfo &setDeviceType(const std::string &device_type);

        UserInfo &setModel(const std::string &vendor, const std::string &name);

        UserInfo &setApplication(const std::string &name, const std::string &version);

        UserInfo &setScreenSize(int width, int height);

        UserInfo &setScreenDpi(int dpi);

        using ExtendedParamsComparatorType = boost::function<bool(const std::string &lhs, const std::string &rhs)>;

        UserInfo &setExtendedParam(const std::string &name, const std::string &value,
                                   UserInfo::ExtendedParamsComparatorType comparator);

        bool matchesConditions(const ItemLocalization::EnableConditions &conditions, const ItemOptions &options) const;

        std::string toString() const;

    private:
        boost::optional<std::string> language_; // part of locale
        boost::optional<std::string> country_;  // part of locale

        boost::optional<std::vector<int> > region_ids_;
        boost::optional<std::vector<int> > region_ids_init_;

        std::map<unsigned long, unsigned long> clids_;

        boost::optional<std::string> uuid_;
        mutable boost::optional<double> uuid_audience_ratio_; // member for lazy calculation
        mutable boost::optional<double> uuid_salted_audience_ratio_; // member for lazy calculation with salt
        mutable std::string last_used_salt_; // last used salt

        boost::optional<std::string> device_type_;
        boost::optional<std::string> vendor_;
        boost::optional<std::string> model_;
        boost::optional<std::string> application_name_;
        boost::optional<std::string> application_version_;
        boost::optional<int> screen_width_;
        boost::optional<int> screen_height_;
        boost::optional<int> screen_dpi_;

        struct MatchExtParam {
            std::string value;
            ExtendedParamsComparatorType comparator;
        };
        using MatchExtendedParams = std::map<std::string, MatchExtParam>;
        MatchExtendedParams match_ext_params_;

        bool uuidMatchesCondition(const ShmStringVector &uuids) const;

        bool localeMatchesCondition(const ShmString &conditional_language, const ShmString &conditional_country) const;

        bool regionIdsMatchesCondition(const ShmUIntSet &localization_region_ids,
                                       const ShmUIntSet &localization_region_ids_blacklist) const;

        bool regionIdsInitMatchesCondition(const ShmUIntSet &localization_region_ids_init,
                                           const ShmUIntSet &localization_region_ids_init_blacklist) const;

        bool audienceRatioMatchesCondition(double audience_ratio, double audience_offset,
                                           const boost::optional<ShmString> &audience_salt) const;

        bool deviceTypeMatchesCondition(const ShmStringVector &conditional_device_types) const;

        bool modelMatchesCondition(const ItemLocalization::EnableConditions::Models &conditional_models) const;

        bool applicationMatchesCondition(const ItemLocalization::EnableConditions::Applications &applications) const;

        bool clidsMatchesCondition(const ItemLocalization::EnableConditions::Clids &clids) const;

        bool extendedParamsMatchesCondition(
                const ItemLocalization::EnableConditions::ExtendedParams &conditional_ext_params) const;

        static bool isGeoTargetingSuccessed(const boost::optional <std::vector<int>> &region_ids,
                                            const ShmUIntSet &localization_region_ids,
                                            const ShmUIntSet &localization_region_ids_blacklist);
    };

}  // end of shared_localization namespace
