import logging
from zipfile import ZipFile

import re
from django.conf import settings
from django.core.management.base import BaseCommand
from yaphone.dialer.dialer.mds import mds_s3_client, MDSError
from yaphone.dialer.dialer.serializers import COMPANY_ICON_URI_TEMPLATE, PHOTO_IMAGE_SIZE, THUMBNAIL_IMAGE_SIZE, COMPANY_ICON_FILENAME_PREFIX

logger = logging.getLogger(__name__)

logging.getLogger("requests").setLevel(logging.WARNING)


def update_organization_icons(zip_file):
    logger.info('Updating icons')

    bucket = mds_s3_client.get_bucket(settings.S3_MDS_BUCKET_NAME)
    s3_keys = {k.key: k for k in bucket.get_all_keys(prefix=COMPANY_ICON_FILENAME_PREFIX)}

    icon_regexp = re.compile(r'launcher-dialer-\w+/organization-icons/(?P<size>photo|thumb)/(?P<category>\w+).png$')
    for filename in zip_file.namelist():
        matched = icon_regexp.match(filename)
        if matched:
            category = matched.groupdict()['category'].lower()
            size = {'thumb': THUMBNAIL_IMAGE_SIZE, 'photo': PHOTO_IMAGE_SIZE}[matched.groupdict()['size']]
            mds_filename = COMPANY_ICON_URI_TEMPLATE.format(category_class=category, size=size)
            try:
                key = s3_keys.pop(mds_filename, None) or bucket.new_key(mds_filename)
                with zip_file.open(filename) as image_file:
                    key.set_contents_from_string(image_file.read(), replace=True, headers={'Content-Type': 'image/png'})
                logger.debug('Saving %s as %s', filename, mds_filename)
            except MDSError:
                logger.error('Could not save icon to mds, filename: %s', filename)
                raise
    if s3_keys:
        try:
            logger.info('Removing old icons')
            bucket.delete_keys(s3_keys.values())
        except MDSError:
            logger.error('Could not remove old files: %s', s3_keys.keys())


class Command(BaseCommand):
    help = 'Updates Dialer resources'

    def add_arguments(self, parser):
        parser.add_argument('--file', type=str, required=True, help=u'zip-archive with resources')

    # noinspection PyAssignmentToLoopOrWithParameter
    def handle(self, *args, **options):
        file_name = options['file']

        with open(file_name) as f:
            zip_file = ZipFile(f)

            update_organization_icons(zip_file)
