from distutils.version import LooseVersion
import logging
import re

import six

from yaphone.utils.django.exceptions import BadRequestAPIError

logger = logging.getLogger(__name__)

DEVICE_PHONE = 'phone'
DEVICE_TABLET = 'tablet'
USER_AGENT_REGEXP = re.compile(r"(?P<app_name>.+)/(?P<app_version_string>.+)\s+"
                               r"\((?P<device_manufacturer>\S+)\s+(?P<device_model>.*);\s+"
                               r"(?P<os_name>.+)\s+(?P<os_version>.*)\)\s*"
                               r"(?P<device_type>(?i)({}|{}))?".format(DEVICE_PHONE, DEVICE_TABLET))


class Locale(object):
    class WrongFormat(BadRequestAPIError):
        default_detail = 'Bad locale format. Locale must fit POSIX standard.'

    # locale format [language[_territory][.codeset][@modifier]]
    # example: ru_RU.UTF-8, en_US, br, de_AT.UTF-8@something
    REGEXP = re.compile(r"(?P<language>[a-z]{2})(_(?P<territory>[A-Z]{2})"
                        r"(\.(?P<codeset>[^@]+)(@(?P<modifier>\w+)?)?)?)?")
    language = None
    territory = None
    codeset = None
    modifier = None

    def __str__(self):
        res = self.language
        if self.territory:
            res = '{}_{}'.format(res, self.territory)
        if self.codeset:
            res = '{}.{}'.format(res, self.codeset)
        if self.modifier:
            res = '{}@{}'.format(res, self.modifier)
        return res

    def __init__(self, *args, **kwargs):
        # locale object can be initialized with string:  Locale("ru_RU")
        # noinspection PyTypeChecker
        if args and isinstance(args[0], six.string_types):
            match = Locale.REGEXP.match(args[0])
            if match is None:
                raise Locale.WrongFormat()
            kwargs = match.groupdict()

        for name, value in kwargs.items():
            setattr(self, name, value)

    def __repr__(self):
        return "<Locale object: %s>" % str(self)


class UserAgent():
    app_name = None
    app_version_string = None
    device_model = None
    device_manufacturer = None
    device_type = None
    os_name = None
    os_version = None
    raw = None

    defective_versions = ('1.5.2', '1.5.3')
    compare_version_len = len(defective_versions[0])

    def __init__(self, app_name, app_version_string, device_type=DEVICE_PHONE, **kwargs):
        if self.is_defective_ua(app_name, app_version_string):
            app_version_string = self.fix_version(app_version_string)
        if device_type is not None:
            device_type = device_type.lower()
        self.app_name = app_name
        self.app_version_string = app_version_string
        self.device_type = device_type
        for k, v in kwargs.items():
            setattr(self, k, v)

    def is_defective_version(self, version):
        return version[:self.compare_version_len] in self.defective_versions

    def is_defective_ua(self, name, version):
        return name == 'com.yandex.launcher' and self.is_defective_version(version)

    def fix_version(self, version):
        return '.'.join((version[:self.compare_version_len], version[self.compare_version_len:]))

    def __str__(self):
        return '%s/%s (%s %s; %s %s)' % (
            self.app_name, self.app_version, self.device_manufacturer, self.device_model, self.os_name, self.os_version,
        )

    @classmethod
    def from_string(cls, ua_str):
        match = USER_AGENT_REGEXP.match(ua_str)
        if match is not None:
            return cls(raw=ua_str, **match.groupdict())

    @property
    def app_version(self):
        return LooseVersion(self.app_version_string)
