import logging

from django.conf import settings
from cache_memoize import cache_memoize
from rest_framework.fields import (Field, FloatField, IntegerField, CharField, SerializerMethodField,
                                   URLField, EmailField, ListField, ChoiceField)
from rest_framework.serializers import Serializer, ListSerializer

from yaphone.dialer.dialer.mds import mds_s3_client
from yaphone.dialer.dialer.models import Locale
from yaphone.dialer.dialer.utils import make_s3_mds_url

COMPANY_ICON_FILENAME_PREFIX = 'dialer_company'
COMPANY_ICON_URI_TEMPLATE = 'dialer_company/{category_class}/{size}'
DEFAULT_CATEGORY_CLASS = 'unknown'
PHOTO_IMAGE_SIZE = 'XL'  # 1280 px by large side
THUMBNAIL_IMAGE_SIZE = 'S'  # 150 px by large side

logger = logging.getLogger(__name__)


class LocaleField(Field):
    def to_representation(self, value):
        return str(value)

    def to_internal_value(self, data):
        return Locale(data)


# noinspection PyAbstractClass
class OrganizationSearchQuery(Serializer):
    lat = FloatField(min_value=-90, max_value=90, required=False)
    lon = FloatField(min_value=-180, max_value=180, required=False)
    count = IntegerField(min_value=1, max_value=50, default=5, required=False)
    query = CharField(min_length=2, max_length=200, required=True)
    language = LocaleField(required=True)


# noinspection PyAbstractClass
class OrganizationLogoQuery(Serializer):
    id = IntegerField(required=True)
    size = ChoiceField(required=True, choices=('photo', 'thumbnail'))
    language = LocaleField(required=True)


# noinspection PyAbstractClass
class Phone(Serializer):
    number = SerializerMethodField()
    info = CharField(required=False)
    type = ChoiceField(required=True, choices=('phone', 'fax', 'phone_fax'))

    @staticmethod
    def get_number(obj):
        """ Returns phone in '+71234567890 format'
        examples of obj :
        {'country': '8',
         'formatted': '8 (800) 250-96-39',
         'number': '2509639',
         'prefix': '800',
         'type': 'phone'},
        {'country': u'7',
         'formatted': u'+7 (495) 739-70-00',
         'info': 'Some info',
         'number': u'7397000',
         'prefix': u'495',
         'type': u'phone'},
        """
        is_plussed = obj['formatted'].strip().startswith('+')
        digits = [d for d in obj['formatted'] if d.isdigit()]

        if not is_plussed and digits[0] == '8' and len(digits) == 11:
            digits[0] = '7'
            is_plussed = True
        number = ''.join(digits)
        if is_plussed:
            number = '+' + number
        return number

    def to_representation(self, data):
        data['type'] = data['type'].lower()
        return super(Phone, self).to_representation(data)


@cache_memoize(timeout=10 * 60)
def get_existing_category_icons():
    bucket = mds_s3_client.get_bucket(settings.S3_MDS_BUCKET_NAME)
    return [k.key for k in bucket.get_all_keys(prefix=COMPANY_ICON_FILENAME_PREFIX)]


# noinspection PyAbstractClass
class OrganizationLogo(Serializer):
    photo = SerializerMethodField()
    thumbnail = SerializerMethodField()

    def get_category_icon(self, obj, size):
        for category in obj.get('category', []):
            if 'class' in category:
                category_class = category['class'].lower().replace(' ', '_')
                icon = COMPANY_ICON_URI_TEMPLATE.format(category_class=category_class, size=size)
                if icon not in self.context['existing_category_icons']:
                    logger.warning('Category "%s" has no icon for size "%s"', category_class, size)
                    continue
                return icon
        return COMPANY_ICON_URI_TEMPLATE.format(category_class=DEFAULT_CATEGORY_CLASS, size=size)

    def get_image_url(self, obj, size):
        if 'existing_category_icons' not in self.context:
            self.context['existing_category_icons'] = set(get_existing_category_icons())

        return make_s3_mds_url(self.get_category_icon(obj, size), self.context.get('host'))

    def get_photo(self, obj):
        return self.get_image_url(obj, PHOTO_IMAGE_SIZE)

    def get_thumbnail(self, obj):
        return self.get_image_url(obj, THUMBNAIL_IMAGE_SIZE)


class OrganizationURLSField(URLField):
    def to_representation(self, data):
        url = None
        for item in data:
            if not url or item['type'] == 'SELF':
                url = item['link']['href']
        return super(OrganizationURLSField, self).to_representation(url)


# noinspection PyAbstractClass
class Organization(OrganizationLogo):
    id = IntegerField(required=True)
    name = CharField(required=True)
    address = CharField(source='address.formatted_address', required=False)
    note = SerializerMethodField(required=False)
    emails = ListField(EmailField(), source='email', required=False)  # absent from protobuf API
    phones = Phone(many=True, source='phone', required=False)
    url = OrganizationURLSField(source='link', required=False)

    @staticmethod
    def get_note(obj):
        categories = [category['name'] for category in obj['category']]
        return '; '.join(categories)


# noinspection PyAbstractClass
class OrganizationSearchResults(ListSerializer):
    child = Organization()

    def to_representation(self, data):
        """
        Skip organizations without phones
        """
        representation = []
        for item in data:
            item_representation = self.child.to_representation(item)
            if 'phones' in item_representation:
                representation.append(item_representation)
        return representation


# noinspection PyAbstractClass
class OrganizationLogoResults(ListSerializer):
    child = OrganizationLogo()

    def to_representation(self, data):
        if data:
            return [self.child.to_representation(item) for item in data]
        else:  # if no organization found, pass empty category list to get default images (DEFAULT_CATEGORY_CLASS)
            return [self.child.to_representation({})]


# noinspection PyAbstractClass
class LocationSerializer(Serializer):
    latitude = FloatField(required=True, min_value=-90, max_value=90)
    longitude = FloatField(required=True, min_value=-180, max_value=180)


# noinspection PyAbstractClass
class GsmSerializer(Serializer):
    country_code = IntegerField(required=True, source='countrycode')
    operator_id = IntegerField(required=True, source='operatorid')
    cell_id = IntegerField(required=True, source='cellid')
    lac = IntegerField(required=True)
    signal_strength = IntegerField(required=True)
    age = IntegerField(required=False, default=0)


# noinspection PyAbstractClass
class WifiSerializer(Serializer):
    mac = CharField(required=True)
    signal_strength = IntegerField(required=True)
    age = IntegerField(required=False, default=0)


# noinspection PyAbstractClass
class GetLocationSerializer(Serializer):
    location = LocationSerializer(required=False)
    cells = GsmSerializer(many=True, required=False)
    wifi_networks = WifiSerializer(many=True, required=False)
