import json
import logging

import requests
from django.conf import settings
from django.http import HttpResponse
from django.views.decorators.http import require_GET
from google.protobuf.json_format import MessageToJson
from requests.exceptions import RequestException, BaseHTTPError
from rest_framework import response as rest_response, views
# noinspection PyUnresolvedReferences
from yandex.maps.proto.common2.response_pb2 import Response as SearchResponse
# noinspection PyUnresolvedReferences
from yandex.maps.proto.search.business_pb2 import GEO_OBJECT_METADATA as BUSINESS_OBJECT

from yaphone.dialer.dialer import exceptions, tvm, utils
from yaphone.utils.django import exceptions as django_exceptions, get_http_header

logger = logging.getLogger(__name__)


@require_GET
def ping(request):
    return HttpResponse('Ok')


class RequestValidatorMixin(object):
    validator_class = None

    def get_validated_data(self, request_data):
        validator = self.validator_class(data=request_data)
        validator.is_valid(raise_exception=True)
        return validator.validated_data


class BaseSearchMapsView(views.APIView):
    query_serializer = None
    result_serializer = None
    tvm = tvm.get_tvm_client()

    @staticmethod
    def external_request(url, params, error):
        try:
            response = requests.get(url, params=params)
            if response.status_code != 200:
                try:
                    message = response.json()['error']
                except (ValueError, TypeError, KeyError):
                    message = response.raw
                raise error(detail=message)
            result = response.json()
        except (ValueError, TypeError, RequestException, BaseHTTPError) as e:
            raise error() from e
        return result

    def initial(self, request, *args, **kwargs):
        super().initial(request, *args, **kwargs)

        user_agent = utils.get_user_agent(request)
        utils.add_log_context(
            app=user_agent.app_name,
            app_version=user_agent.app_version_string,
            device='{0.device_manufacturer} {0.device_model}'.format(user_agent),
            os='{0.os_name} {0.os_version}'.format(user_agent),
            request_id=get_http_header(request, 'X-Request-Id'),
        )

    def get(self, request):
        serializer = self.query_serializer(data=request.query_params)
        serializer.is_valid(raise_exception=True)
        self.data = serializer.validated_data

        # Maps search backend API is described here:
        # https://wiki.yandex-team.ru/maps/api/http/internal/search/guide/
        search_query = self.get_search_query(self.data, request)
        try:
            headers = tvm.get_service_ticket_headers(self.tvm, settings.MAPS_SEARCH_CLIENT_ID)
        except tvm.TvmTicketCheckError:
            logger.exception("Error preparing maps search TVM ticket")
            raise exceptions.MapsSearchRequestException()

        try:
            response = requests.get(settings.MAPS_SEARCH_PROTO_API_URL, headers=headers, params=search_query, timeout=2)
            if response.status_code == 400:
                raise django_exceptions.BadRequestAPIError()
            response.raise_for_status()
        except (RequestException, BaseHTTPError):
            logger.exception("HTTP error maps search request")
            raise exceptions.MapsSearchBackendUnavailable()

        try:
            proto = SearchResponse()
            proto.ParseFromString(response.content)
        except Exception:
            logger.exception("Error parsing maps search response")
            raise exceptions.MapsSearchRequestException()

        companies = []
        for reply in proto.reply.geo_object:
            for obj in reply.metadata:
                if obj.HasExtension(BUSINESS_OBJECT):
                    company = obj.Extensions[BUSINESS_OBJECT]
                    # MessageToJson + json.loads should be just MessageToDict
                    # if only it could convert all string fields to unicode
                    companies.append(json.loads(MessageToJson(company, preserving_proto_field_name=True)))

        serializer = self.result_serializer(companies, context={'request': request, 'host': request.get_host()})
        self.fill_spam_info(serializer.data)
        return rest_response.Response(serializer.data)

    def get_search_query(self, data, request):
        raise NotImplementedError

    def fill_spam_info(self, organizations):
        """
        Fill spam feature from UGC database using Caller ID API
        https://wiki.yandex-team.ru/JandeksPoisk/Antispam/whocallsya/backend/#info1
        """
        phones = set()
        for organization in organizations:
            for phone in organization['phones']:
                phones.add(phone['number'])
        if not phones:
            # Nothing to do here
            return
        params = {
            'phonenums': ','.join(phones),
            'lang': self.data['language'].language,
            'allphones': 'false',
            'verticals': 'ugc',
        }
        try:
            response = requests.get(settings.CALLER_ID_INFO_URL, params=params)
            if response.status_code != 200:
                try:
                    message = response.json()['error']
                except (ValueError, TypeError, KeyError):
                    message = response.raw
                raise exceptions.CallerIDBackendUnavailable(detail=message)
            result = response.json()
        except Exception:
            logger.exception('CallerID exception')
            raise
        ugc_map = {item['phonenum']: item['result']['ugc'] for item in result}
        for organization in organizations:
            for phone in organization['phones']:
                ugc = ugc_map.get(phone['number'])
                is_spam = ugc is not None and ugc['polarity'] == 'negative'
                phone['spam'] = is_spam
                if is_spam:
                    phone['spam_reason'] = ugc['description']
