import logging
from uuid import UUID

from rest_framework.exceptions import NotFound
from rest_framework.response import Response
from rest_framework.views import APIView
from yaphone.http_geobase_lookuper.httpgeobase import FED_SUBJECT_TYPE
from yaphone.utils.django import get_http_header, get_user_ip
from yaphone.utils.geo import geobase_lookuper, get_parent_region_with_type, lbs_locator
from yaphone.utils.django.exceptions import BadRequestAPIError
from yaphone.dialer.dialer import serializers, views

logger = logging.getLogger(__name__)


class GetLocationView(APIView, views.base.RequestValidatorMixin):
    validator_class = serializers.GetLocationSerializer

    @staticmethod
    def get_uuid(request):
        uuid_string = get_http_header(request, 'X_YAUUID')
        if not uuid_string:
            raise BadRequestAPIError("X-YAUUID header is missing")
        try:
            return UUID(uuid_string)
        except ValueError:
            raise BadRequestAPIError("Badly formed hexadecimal UUID string [%s]" % uuid_string)

    def initial(self, request, *args, **kwargs):
        super(GetLocationView, self).initial(request, *args, **kwargs)
        self.ip = get_user_ip(request)
        self.uuid = self.get_uuid(request)

    def get_lbs_location(self, data):
        lbs_position = lbs_locator.locate(
            gsm_cells=data.get('cells', []),
            wifi_networks=data.get('wifi_networks', []),
            uuid=self.uuid,
            ip=self.ip
        )
        if not lbs_position or lbs_position['type'] == 'ip':
            raise NotFound(detail='Could not determine user location')
        return lbs_position['latitude'], lbs_position['longitude'], lbs_position['precision']

    def post(self, request, *args, **kwargs):
        data = self.get_validated_data(request.data)

        if data.get('location'):
            location = data['location']
            latitude, longitude, accuracy = location['latitude'], location['longitude'], 0.
        else:
            latitude, longitude, accuracy = self.get_lbs_location(data)

        region_id = geobase_lookuper.get_region_id_by_location(latitude, longitude)
        region = geobase_lookuper.get_region_by_id(region_id)

        return Response({
            'region': {
                'name': region['name'],
                'region_id': region_id,
                'subject_id': get_parent_region_with_type(region_id, FED_SUBJECT_TYPE),
            },
            'position': {
                'lat': latitude,
                'lon': longitude,
                'accuracy': accuracy,
            },
        })
