import logging

from django import http
from rest_framework import exceptions

from yaphone.utils.django import get_user_ip
from yaphone.utils.geo import geobase_lookuper

from yaphone.dialer.dialer import serializers, views

logger = logging.getLogger(__name__)


class OrganizationLogoView(views.base.BaseSearchMapsView):
    query_serializer = serializers.OrganizationLogoQuery
    result_serializer = serializers.OrganizationLogoResults

    def get_search_query(self, data, request):
        return {
            'business_oid': data['id'],
            'lang': data['language'].language,
            'ms': 'pb',  # protobuf format
            # 'hr': 'yes',  # human-readable protobuf, for debuging
            'snippets': 'businessimages/1.x',  # logos
            'results': 1,
        }

    def fill_spam_info(self, organizations):
        # Not needed here
        pass

    def get(self, request):
        response = super(OrganizationLogoView, self).get(request)
        try:
            image_url = response.data[0][self.data['size']]
        except IndexError:
            raise exceptions.NotFound('Logo not found')
        return http.HttpResponseRedirect(image_url)


class OrganizationSearchView(views.base.BaseSearchMapsView):
    query_serializer = serializers.OrganizationSearchQuery
    result_serializer = serializers.OrganizationSearchResults

    def get_search_query(self, data, request):
        search_query = {
            'text': data['query'],
            'lang': data['language'].language,
            'results': data['count'],
            'type': 'biz',  # Organizations only search
            'ms': 'pb',
            # 'hr': 'yes',
            'snippets': 'businessimages/1.x',  # logos
        }
        if 'lat' in data and 'lon' in data:
            # use narrow search window to force organization nearby search
            # https://st.yandex-team.ru/PHONE-975
            search_query['ull'] = search_query['ll'] = '%(lon).6f,%(lat).6f' % data
            search_query['spn'] = '0.05,0.05'
        else:
            ip = get_user_ip(request)
            if ip:
                try:
                    region = geobase_lookuper.get_region_by_ip(ip)
                    search_query['experimental_lr'] = region['id']
                except RuntimeError:
                    pass
        return search_query
