package main

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"regexp"
	"strings"

	"a.yandex-team.ru/yaphone/gotifier/internal/config"
)

func main() {
	http.HandleFunc("/qloud", QloudHandler)
	log.Printf("Listening on port: %d", config.Port)
	err := http.ListenAndServe(fmt.Sprintf(":%d", config.Port), nil)
	if err != nil {
		log.Fatal("ListenAndServe: ", err)
	}
}

func QloudHandler(w http.ResponseWriter, r *http.Request) {
	request, err := ParseRequest(r)
	if err != nil {
		log.Printf("Error parsing request: %v", err)
		http.Error(w, "Bad request body", http.StatusBadRequest)
		return
	}

	if _, ok := config.QloudNotificationStatuses[request.Status]; !ok {
		log.Printf("Ignore status %v", request.Status)
		return
	}

	splittedEnv := strings.Split(request.EnvironmentID, ".")
	message := fmt.Sprintf("**%s %s** %s.", splittedEnv[1], splittedEnv[2], request.Status)
	buttons := []map[string]string{GetQloudButton(request.EnvironmentID)}
	AppendButtons(request.Comment, &buttons, &message)
	for _, button := range buttons {
		message += fmt.Sprintf("\n[%s](%s)", button["text"], button["url"])
	}

	body := QRequestBody{
		ChatID: config.QChatID,
		Text:   message,
	}

	serializedBody, err := json.Marshal(body)
	if err != nil {
		log.Printf("Failed to serialize request body: %v", err)
	}
	auth := fmt.Sprintf("OAuthTeam %s", config.QToken)
	req, err := http.NewRequest("POST", config.QURL, bytes.NewBuffer(serializedBody))
	if err != nil {
		log.Printf("Creating request to Q API failed, %v", err)
		return
	}
	req.Header.Add("Authorization", auth)
	req.Header.Add("Content-Type", "application/json")
	client := &http.Client{}
	_, err = client.Do(req)
	if err != nil {
		log.Printf("Request to Q API failed, %v", err)
	}
}

type QloudRequest struct {
	Status        string `json:"status"`
	EnvironmentID string `json:"environmentId"`
	Comment       string `json:"comment"`
}

type QRequestBody struct {
	ChatID string `json:"chat_id"`
	Text   string `json:"text"`
}

func ParseRequest(r *http.Request) (data QloudRequest, err error) {
	body, err := ioutil.ReadAll(r.Body)
	if err != nil {
		return
	}
	log.Printf("Got request: %v", string(body))
	err = json.Unmarshal(body, &data)
	return
}

func GetQloudButton(environmentID string) map[string]string {
	url := fmt.Sprintf(config.QloudEnvironmentURLTemplate,
		strings.Join(strings.Split(environmentID, "."), "/"))
	return map[string]string{
		"text": "Qloud",
		"url":  url,
	}
}

func AppendButtons(comment string, buttons *[]map[string]string, message *string) {
	re := regexp.MustCompile(`:(\d{7,10})[,}].*$`)
	revision := re.FindStringSubmatch(comment)
	if len(revision) == 0 {
		*message += fmt.Sprintf("\nDeploy comment: %s", comment)
		return
	}
	arcButton := map[string]string{
		"text": "Arcanum",
		"url":  fmt.Sprintf(config.ArcURLTemplate, revision[1]),
	}
	*buttons = append(*buttons, arcButton)

	if commitMessage, ok := GetRevisionMessage(revision[1]); ok {
		*message += fmt.Sprintf("\n**%s**: __'%s'__",
			commitMessage["author"],
			strings.Split(commitMessage["message"].(string), "\n")[0])
		if STButton, ok := GetSTButton(commitMessage["message"].(string)); ok {
			*buttons = append(*buttons, STButton)
		}
	}
}

func GetRevisionMessage(revision string) (data map[string]interface{}, ok bool) {
	url := fmt.Sprintf(config.ArcCommitURLTemplate, revision)
	auth := fmt.Sprintf("OAuth %s", config.ArcanumToken)
	req, err := http.NewRequest("GET", url, nil)
	if err != nil {
		log.Printf("Error occured before Arcanum request: %v", err)
		return
	}
	req.Header.Add("Authorization", auth)
	client := &http.Client{}
	resp, err := client.Do(req)
	if resp != nil {
		defer func() {
			if err := resp.Body.Close(); err != nil {
				log.Printf("Error occured while closing Arcanum connection: %v", err.Error())
			}
		}()
	}
	if err != nil {
		log.Printf("Error occured while Arcanum request: %v", err)
		return
	}

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		log.Printf("Error occured while reading Arcanum response: %v", err)
		return
	}
	log.Printf(`Got Arcanum response: %s`, body)
	if resp.StatusCode != 200 {
		log.Printf("Bad Arcanum response code: %v", resp.StatusCode)
		return
	}
	err = json.Unmarshal(body, &data)
	if err != nil {
		log.Printf("Error occured while parsing Arcanum response: %s", body)
		return
	}
	ok = true
	return
}

func GetSTButton(message string) (button map[string]string, ok bool) {
	re := regexp.MustCompile(`[A-Z]{3,}-\d+`)
	ticket := re.FindString(message)
	if len(ticket) == 0 {
		return
	}
	ok = true
	button = map[string]string{
		"text": ticket,
		"url":  fmt.Sprintf(config.StURLTemplate, ticket),
	}
	return
}
