package yenv

import (
	"fmt"
	"io/ioutil"
	"os"
)

var fallbacks = map[string][]string{
	"development": {"testing", "prestable", "production"},
	"unstable":    {"testing", "prestable", "production"},
	"testing":     {"prestable", "production"},
	"prestable":   {"production"},
	"stress":      {"testing", "production"},
	"datatesting": {"other"},
}

func choose(available map[string]interface{}, value string, fallback bool) interface{} {
	for env := range available {
		if env == value {
			return available[env]
		}
	}
	if fallback {
		for _, env := range fallbacks[value] {
			for availableEnv := range available {
				if env == availableEnv {
					return available[env]
				}
			}
		}
	}
	panic("no environment is available")
}

func load(suffix string, default_ string) (value string) {
	varName := fmt.Sprintf("YENV_%s", suffix)
	value = os.Getenv(varName)
	if len(value) == 0 {
		value = default_
		dat, err := ioutil.ReadFile(fmt.Sprintf("/etc/yandex/environment.%s", suffix))
		if err == nil {
			value = string(dat)
		}
		err = os.Setenv(varName, value)
		if err != nil {
			panic(err)
		}
	}
	return
}

func ChooseKeyByType(d map[string]interface{}, fallback bool) interface{} {
	return choose(d, Type, fallback)
}

func ChooseKeyByName(d map[string]interface{}, fallback bool) interface{} {
	return choose(d, Name, fallback)
}

var Type = load("TYPE", "development")
var Name = load("NAME", "localhost")
