# -*- coding: utf-8 -*-
import logging
import requests

logger = logging.getLogger(__name__)

DEFAULT_URL = 'http://geobase.qloud.yandex.ru'

# Geobase constants https://doc.yandex-team.ru/lib/libgeobase5/concepts/region-types.xml
EARTH_REGION_ID = 10000
CONTINENT_TYPE = 1  # Континент
REGION_TYPE = 2  # Регион типа Европейский союз
COUNTRY_TYPE = 3  # Страна
FED_OKRUG_TYPE = 4  # Федеральный округ.
FED_SUBJECT_TYPE = 5  # Субъект федерации.
CITY_TYPE = 6  # Город.
VILLAGE_TYPE = 7  # Село.
CITY_REGION_TYPE = 8  # Район города.
METRO_STATION_TYPE = 9  # Станция метро.
FEDERATION_SUBJECT_REGION_TYPE = 10  # Район субъекта федерации.
AIRPORT_TYPE = 11  # Аэропорт.
OVERSEAS_TYPE = 12  # Заморская территория.
SMALL_CITY_REGION_TYPE = 13  # Район города второго уровня.
MONORAIL_STATION_TYPE = 14  # Станция монорельса.
SUBURBAN_TYPE = 15  # Сельское поселение.


class HttpGeobaseException(RuntimeError):
    """ RuntimeError is used to be compatible with the old geobase behavior """


class _GeobaseObject(object):
    """ Wraps dict returned by HTTP API. Allows access to keys as attributes. """

    def __init__(self, data):
        assert isinstance(data, dict)
        self._data = data

    def __getattr__(self, item):
        if item not in self._data:
            raise AttributeError("'%s' object has no attribute '%s'" % (self.__class__.__name__, item))
        return self._data[item]

    def __repr__(self):
        return repr(self._data)


class Linguistics(_GeobaseObject):
    cases = {'ablative', 'accusative', 'dative', 'directional',
             'genitive', 'instrumental', 'locative', 'nominative', 'prepositional'}

    def __dir__(self):
        return list(self.cases) + ['preposition']

    def __getattr__(self, item):
        if item in self.cases:
            item += '_case'
        return super(Linguistics, self).__getattr__(item)


class Region(_GeobaseObject):
    @staticmethod
    def __dir__(*args, **kwargs):
        return [
            'capital_id', 'en_name', 'geo_parent_id', 'id', 'is_main', 'iso_name',
            'lat', 'latitude_size', 'lon', 'longitude_size', 'name', 'native_name',
            'official_languages', 'parent_id', 'phone_code', 'population',
            'position', 'services', 'short_en_name', 'synonyms', 'type',
            'tzname', 'widespread_languages', 'zip_code', 'zoom', 'as_dict',
        ]

    @property
    def as_dict(self):
        return self._data

    @property
    def lat(self):
        return self._data['latitude']

    @property
    def lon(self):
        return self._data['longitude']


class Timezone(_GeobaseObject):
    @staticmethod
    def __dir__(*args, **kwargs):
        return ['abbr', 'dst', 'name', 'offset']


class Lookup(object):
    """
    Implements methods of geobase.Lookup.
    """
    api_version = 1

    # TODO: List of not implemented methods from geobase5.Lookup:
    # as_by_ip
    # get_border_points
    # get_known_timezones
    # get_known_timezones_for_lang
    # pinpoint_geolocation, pinpointGeolocation
    # tree

    def __init__(self, url=DEFAULT_URL):
        self.base_url = '{url}/v{api_version}'.format(url=url, api_version=self.api_version)

    def _make_request(self, method, params):
        try:
            response = requests.get('{base_url}/{method}'.format(
                base_url=self.base_url,
                method=method),
                params=params
            )
            response.raise_for_status()
            return response.json()
        except (requests.exceptions.BaseHTTPError, requests.exceptions.RequestException) as e:
            raise HttpGeobaseException("Couldn't make a request to http-geobase: {}".format(e))

    def parents(self, region_id):
        return self._make_request('parents', {'id': region_id})

    def children(self, region_id):
        return self._make_request('children', {'id': region_id})

    def region_by_id(self, region_id):
        return Region(self._make_request('region_by_id', {'id': region_id}))

    def region_id_by_location(self, lat, lon):
        return self._make_request('region_id_by_location', {'lat': lat, 'lon': lon})

    def find_country_id(self, region_id):
        return self._make_request('find_country', {'id': region_id})

    def subtree(self, region_id):
        return self._make_request('subtree', {'id': region_id})

    def regions_by_type(self, region_type):
        return map(Region, self._make_request('regions_by_type', {'type': region_type}))

    def id_is_in(self, region_id, parent_region_id):
        return self._make_request('in', {'pid': parent_region_id, 'id': region_id})

    def linguistics(self, region_id, language):
        return Linguistics(self._make_request('linguistics_for_region', {'lang': language, 'id': region_id}))

    def region_by_ip(self, ip):
        return Region(self._make_request('region_by_ip', {'ip': ip}))

    def region_id(self, ip):
        return self._make_request('region_id', {'ip': ip})

    def asset(self, ip):
        return self._make_request('asset', {'ip': ip})

    def known_linguistics(self):
        return self._make_request('supported_linguistics', {})

    def get_timezone(self, region_id):
        return self._make_request('timezone', {'id': region_id})

    def tzinfo(self, name):
        return Timezone(self._make_request('tzinfo', {'name': name}))

    def timezone_by_id(self, region_id):
        return self.tzinfo(self.get_timezone(region_id))

    def reliabilities_by_ip(self, ip):
        return self._make_request('reliabilities_by_ip', {'ip': ip})

    def is_tor(self, ip):
        return self._make_request('is_tor', {'ip': ip})

    def is_in(self, ip, region_id):
        return self.id_is_in(self.region_id(ip), region_id)

    def regions(self, ip):
        return map(self.region_by_id, self.parents(self.region_by_ip(ip)))

    # Synonyms
    get_country_id = findCountryId = find_country_id
    idIsIn = id_is_in
    regionById = region_by_id
    regionIdByLocation = region_id_by_location
    region = regionByIP = region_by_ip
    regionId = region_id
    regionsByType = regions_by_type
    timezoneById = timezone_by_id
    isIn = is_in

    # geobase6 methods
    def get_region_by_id(self, region_id):
        return self._make_request('region_by_id', {'id': region_id})

    def get_region_by_ip(self, ip):
        return self._make_request('region_by_ip', {'ip': ip})

    get_region_id_by_location = region_id_by_location
    get_parents_ids = parents
    get_linguistics = linguistics
