import hashlib
import logging
import threading

from boto.s3.connection import S3Connection

logger = logging.getLogger(__name__)


class S3Client(object):
    extension_name = 's3mds'
    _thread_data = threading.local()

    class AlreadyExists(Exception):
        pass

    class ConnectionError(Exception):
        pass

    def __init__(self, app=None, config_prefix='S3_MDS'):
        self.config_prefix = config_prefix

        if app is not None:
            self.init_app(app)

    def init_app(self, app):

        self.register_extension(app)

        def config_key(suffix):
            return app.config['%s_%s' % (self.config_prefix, suffix)]

        self.host = config_key('HOST')
        self.bucket_name = config_key('BUCKET_NAME')
        self.access_key_id = config_key('ACCESS_KEY_ID')
        self.access_secret_key = config_key('ACCESS_SECRET_KEY')

    def register_extension(self, app):
        if self.extension_name not in app.extensions:
            app.extensions[self.extension_name] = {}
        if self.config_prefix in app.extensions[self.extension_name]:
            raise Exception('duplicate config_prefix "%s"' % self.config_prefix)
        app.extensions[self.extension_name][self.config_prefix] = self

    def _get_bucket(self):
        try:
            connection = S3Connection(
                host=self.host,
                is_secure=True,
                aws_access_key_id=self.access_key_id,
                aws_secret_access_key=self.access_secret_key,
            )
            return connection.get_bucket(self.bucket_name)
        except Exception as e:
            msg = 'Cannot connect to s3 server: %s' % e
            logger.error(msg)
            raise self.ConnectionError(msg)

    @property
    def bucket(self):
        if hasattr(self._thread_data, 'bucket'):
            return self._thread_data.bucket
        self._thread_data.bucket = self._get_bucket()
        return self._thread_data.bucket

    def get_all_keys(self, prefix):
        return self.bucket.get_all_keys(prefix=prefix)

    def get_key(self, name):
        return self.bucket.get_key(name)

    def new_key(self, name):
        return self.bucket.new_key(name)

    def save(self, data, key_name, metadata=None, replace=False):
        metadata = metadata or {}
        key = self.get_key(key_name)
        if key is not None:
            # Key exists
            md5 = hashlib.md5(data).hexdigest()
            current_md5 = key.etag.strip('"')
            if key.etag and md5 == current_md5:
                # File is the same
                logger.info('S3 object [%s] hash [%s] already exists.', key_name, md5)
                return key
            if not replace:
                msg = 'S3 object [%s] hash [%s] already exists. Cannot replace with [%s]' % (key_name, current_md5, md5)
                logger.warning(msg)
                raise self.AlreadyExists(msg)
        else:
            key = self.bucket.new_key(key_name)
        self.set_key(key, data, metadata)
        return key

    def set_key(self, key, data, metadata=None, md5=None):
        metadata = metadata or {}
        md5 = md5 or hashlib.md5(data).hexdigest()
        key.update_metadata(metadata)
        headers = {'content-type': metadata.get('mimetype', 'application/octet-stream')}
        key.set_contents_from_string(data, md5=key.get_md5_from_hexdigest(md5), headers=headers)
        logger.info('S3 object [%s] hash [%s] has been successfully saved.', key.name, md5[0])
        return key

    def delete(self, key_name):
        key = self.bucket.delete_key(key_name)
        logger.info('Deleted key=%s', key)
        return key

    def delete_many(self, key_names):
        keys = self.bucket.delete_keys(key_names)
        logger.info('Deleted keys=%s', keys.deleted)
        return keys

    def get_url(self, key_name):
        key_name = key_name.encode('utf-8')
        return 'https://{bucket_name}.{host}/{key_name}'.format(
            bucket_name=self.bucket_name,
            host=self.host,
            key_name=key_name
        ).decode('utf-8')
