import json
import shutil
import tempfile
from StringIO import StringIO

import fabric
import os
import re
import requests
from fabric.api import *
from fabric.context_managers import settings
from fabric.exceptions import CommandTimeout
from fabric.operations import local

QLOUD_API_TEMPLATE = 'https://platform.yandex-team.ru/api/v1/environment/overview/{environment}'
QLOUD_TOKEN = os.environ.get('QLOUD_OAUTH_TOKEN')

qloud_components_roles = {
    'regression': ('yandex-phone.jafar.regression', ['recommender']),
    'testing': ('yandex-phone.jafar.testing', ['recommender']),
    'stress': ('yandex-phone.jafar.stress', ['recommender']),
    'prestable': ('yandex-phone.jafar.production', ['recommender-prestable']),
    'stable': ('yandex-phone.jafar.production', ['recommender', 'recommender-prestable']),
    'worker_testing': ('yandex-phone.jafar.testing', ['worker']),
    'worker_stable': ('yandex-phone.jafar.production', ['worker']),
}


def get_hosts_from_qloud(role):
    environment, components = qloud_components_roles[role]
    headers = {"Authorization": "OAuth %s" % QLOUD_TOKEN}
    url = QLOUD_API_TEMPLATE.format(environment=environment)
    response = requests.get(url=url, headers=headers)
    try:
        environment = response.json()
    except Exception as e:
        print 'error: cannot get hosts: %s' % e
        print 'response: %s (%s)' % (response.content, response.status_code)
        raise
    return [
        instance['host']
        for component in components
        for instance in environment['components'][component]['instances'].itervalues()
        ]


# Define roles
env.roledefs = {role: get_hosts_from_qloud(role) for role in qloud_components_roles}

env.roles = env.roles or ['testing']


def continuous(*args, **kwargs):
    try:
        run(*args, **kwargs)
    except KeyboardInterrupt:
        pass


@task
@parallel
def cmd(c):
    run(c)


@task
def make_ammo(experiment, host=None, collection_time=60, output_file='jafar.ammo'):
    from jafar.utils.ammo import make_request

    if not host:
        # get default stress host
        host = env.roledefs['stress'][0]

    print 'Capturing last {} seconds of jafar.log output'.format(collection_time)

    @parallel
    def capture_log():
        io = StringIO()
        try:
            run('tail -F /var/log/yandex/jafar/jafar.log', timeout=collection_time, stdout=io)
        except (KeyboardInterrupt, CommandTimeout):
            pass
        io.seek(0)
        return io

    output = execute(capture_log, hosts=env.hosts)
    regex = re.compile(r'\[jafar.views\]\sRequest: (.*)$')
    count = 0
    with open(output_file, 'w') as f:
        for io in output.values():
            for line in io:
                match = re.findall(regex, line)
                if match:
                    f.write(make_request(experiment, host, match[0]))
                    count += 1

    print 'Prepared {} ammo requests. Results are stored in {}'.format(count, output_file)


# helper functions for `update_snapshot_dev`

def _restore_snapshot_from_deb(package):
    from jafar import create_app
    from jafar.storages.memmap.storage import restore_dump

    directory = tempfile.mkdtemp()
    os.makedirs(os.path.join(directory, 'archives', 'partial'))
    # download package
    local("apt-get -d -o dir::cache={} -o Debug::NoLocking=1 install {}".format(directory, package))
    # find the .deb file
    deb_file = [
        name for name in os.listdir(os.path.join(directory, 'archives'))
        if name.endswith('.deb')
    ][0]
    # extract contents
    local('dpkg-deb -x {}/{} {}'.format(os.path.join(directory, 'archives'), deb_file, directory))
    with open(os.path.join(directory, 'etc', 'jafar-snapshot', 'jafar-snapshot.json')) as f:
        resource = json.load(f)['skynet_resource']
    # download snapshot
    snapshot_path = os.path.join(directory, 'snapshot')
    local('/usr/local/bin/sky get -u -d {} -w {}'.format(snapshot_path, resource))

    # get file path ignoring any intermediate folders
    for root, dirs, files in os.walk(snapshot_path, topdown=False):
        app = create_app()
        with app.app_context():
            restore_dump(os.path.join(root, files[0]))
            break

    shutil.rmtree(directory)


@task
@fabric.decorators.hosts("localhost")
def update_snapshot_dev():
    from jafar import create_app
    from jafar.storages.memmap.storage import restore_dump

    print(
        "\n"
        "What particular kind of snapshot is required?\n"
        "    1) The latest snapshot from worker (if you need the most fresh snapshot)\n"
        "    2) The latest snapshot from deployment installation (if you need the current testing/production version)\n"
        "    3) Specific version of yandex-mobile-jafar-snapshot (if you need some previous testing/production version)\n"
    )
    valid_choices = ('1', '2', '3')
    while True:
        choice = raw_input().strip().lower()
        if choice == 'exit':
            return
        elif choice not in valid_choices:
            print 'Please choose one of {}'.format(valid_choices)
        else:
            break

    if choice == '3':
        print "Enter the debian package name including version (e.g. yandex-mobile-jafar-snapshot=0.1.2-stable-56):"
        while True:
            package_choice = raw_input().strip().lower()
            if package_choice == 'exit':
                return
            if not package_choice.startswith('yandex-mobile-jafar-snapshot='):
                print 'Incorrect package name (must start with "yandex-mobile-jafar-snapshot=")'
            else:
                _restore_snapshot_from_deb(package_choice)
                return

    elif choice in ('1', '2'):
        print 'Choose the environment [testing/stable]:'
        valid_choices = ('testing', 'stable')
        while True:
            environment_choice = raw_input().strip().lower()
            if environment_choice == 'exit':
                return
            if environment_choice not in valid_choices:
                print 'Please choose one of {}'.format(valid_choices)
            else:
                break

        if environment_choice == 'stable':
            print (
                'Warning: stable environment is chosen, network access issues are possible. '
                'Make sure you run this command on a machine that has access to production servers '
                '(for example, your laptop and not your dev server)'
            )

        # if 2, grab the current version of yandex-mobile-jafar-snapshot from the selected environment
        if choice == '2':
            io = StringIO()
            with settings(host_string=env.roledefs[environment_choice][0]):
                run('dpkg -l | grep yandex-mobile-jafar-snapshot || true', stdout=io)
            io.seek(0)
            output = io.read().split()
            if not output:
                raise Exception(
                    "yandex-mobile-jafar-snapshot is not installed in {} environment".format(environment_choice)
                )
            package_name, package_version = output[3], output[4]
            package = '{}={}'.format(package_name, package_version)
            _restore_snapshot_from_deb(package)
            return

        # if 1, no deb package is required, but we have to peek into worker's `snapshots` directory
        else:
            environment_choice = 'worker_' + environment_choice
            snapshot_dir = '/var/lib/jafar/snapshots'

            def snapshots_gen():
                for host in env.roledefs[environment_choice]:
                    io = StringIO()
                    with settings(host_string=host):
                        run('ls {} || true'.format(snapshot_dir), stdout=io)
                    io.seek(0)
                    for name in io.read().split():
                        if name.strip().endswith('tar.gz'):
                            yield host, name

            snapshots = list(snapshots_gen())
            if not snapshots:
                raise Exception("No snapshots found on workers in {} environment".format(environment_choice))

            # take latest snapshot

            def snapshot_order_key(item):
                host, name = item
                _, version, date = name.strip('.tar.gz').split('_')
                return (int(version), date)

            host, file_name = sorted(snapshots, key=snapshot_order_key)[-1]
            directory = tempfile.mkdtemp()
            with settings(host_string=host):
                get(os.path.join(snapshot_dir, file_name), directory)
            app = create_app()
            with app.app_context():
                restore_dump(os.path.join(directory, file_name))
            shutil.rmtree(directory)
            return
