import logging
from gevent.lock import Semaphore
from functools import partial
from clickhouse_driver import Client
from clickhouse_driver.defines import DEFAULT_PORT
from clickhouse_driver.errors import Error as ClickhouseError, ErrorCodes as ClickhouseErrorCodes

logger = logging.getLogger(__name__)


class ConnectionPoolError(ClickhouseError):
    code = ClickhouseErrorCodes.NO_FREE_CONNECTION

class ClickHouse(object):

    param_names = [
        'PORT', 'DATABASE', 'USER', 'PASSWORD', 'CLIENT_NAME', 'CIPHERS',
        'CONNECT_TIMEOUT', 'CA_CERTS', 'SEND_RECEIVE_TIMEOUT', 'VERIFY',
        'PREFIX_COMPRESS_BLOCK_SIZE', 'COMPRESSION', 'SECURE', 'SSL_VERSION',
        'SYNC_REQUEST_TIMEOUT', 'SETTINGS'
    ]

    def __init__(self, app=None, config_prefix='CLICKHOUSE'):
        self.config_prefix = config_prefix
        self.connection_pool = None
        if app is not None:
            self.init_app(app, config_prefix)

    def init_app(self, app, config_prefix='CLICKHOUSE'):
        """Initialize the `app` for use with this :class:`~ClickHouse`. This is
        called automatically if `app` is passed to :meth:`~ClickHouse.__init__`.

        The app is configured according to the configuration variables
        ``PREFIX_HOST``, ``PREFIX_PORT``, ``PREFIX_DATABASE``,
        ``PREFIX_USER``, ``PREFIX_PASSWORD``, ``PREFIX_CLIENT_NAME``,
        ``PREFIX_CONNECT_TIMEOUT``, ``PREFIX_SEND_RECEIVE_TIMEOUT``,
        ``PREFIX_SYNC_REQUEST_TIMEOUT``, ``PREFIX_COMPRESS_BLOCK_SIZE``,
        ``PREFIX_COMPRESSION``, ``PREFIX_SECURE``, ``PREFIX_VERIFY``,
        ``PREFIX_SSL_VERSION``, ``PREFIX_CA_CERTS`` and  ``PREFIX_CIPHERS``,
        where "PREFIX" defaults to "CLICKHOUSE".

        :param flask.Flask app: the application to configure for use with
           this :class:`~ClickHouse`
        :param str config_prefix: determines the set of configuration
           variables used to configure this :class:`~ClickHouse`
        """
        self.config_prefix = config_prefix

        if 'clickhouse' not in app.extensions:
            app.extensions['clickhouse'] = {}

        if config_prefix in app.extensions['clickhouse']:
            raise Exception('duplicate config_prefix "%s"' % config_prefix)

        def key(suffix):
            return '%s_%s' % (config_prefix, suffix)

        app.config.setdefault(key('HOST'), 'localhost')
        app.config.setdefault(key('PORT'), DEFAULT_PORT)
        app.config.setdefault(key('DATABASE'), app.name)
        app.config.setdefault(key('CONNECT_TIMEOUT'), 10)
        app.config.setdefault(key('CONNECTION_POOL'), 100)

        for key_name in ('PORT', 'CONNECT_TIMEOUT', 'CONNECTION_POOL'):
            if not isinstance(app.config[key(key_name)], int):
                raise TypeError('%s_%s must be an integer' % (config_prefix, key_name))

        kwargs = {}

        for param in self.param_names:
            value = app.config.get(key(param))
            if value is not None:
                kwargs[param.lower()] = value

        self.lock_timeout = app.config.get(key('CONNECT_TIMEOUT'))
        self.semaphore = Semaphore(app.config[key('CONNECTION_POOL')])
        self.connection_pool = [
            Client(app.config[key('HOST')], **kwargs)
            for _ in xrange(app.config[key('CONNECTION_POOL')])
        ]

        app.extensions['clickhouse'][config_prefix] = self

    def _execute(self, name, *args, **kwargs):
        if self.semaphore.acquire(timeout=self.lock_timeout):
            try:
                client = self.connection_pool.pop()
                try:
                    return getattr(client, name)(*args, **kwargs)
                finally:
                    self.connection_pool.append(client)
            finally:
                self.semaphore.release()
        else:
            raise ConnectionPoolError("No free connection")

    def __getattr__(self, name):
        return partial(self._execute, name)
