import logging
import logging.config
import warnings
import yenv

import mongoengine
import mongomock
from flask.exthook import ExtDeprecationWarning
from mock import patch

# ignore message "Importing flask.ext.cache is deprecated, use flask_cache instead"
# until flask_cache will be updated
warnings.filterwarnings('ignore', category=ExtDeprecationWarning, module=r'.*flask_cache.*')

# disable accidental logging occurring when some library imports
logging.disable(logging.FATAL)

from flask import Flask
from flask_admin import Admin
from flask_bootstrap import Bootstrap
from flask_cache import Cache
from flask_clickhouse import ClickHouse
from flask_login import LoginManager
from flask_mongoengine import MongoEngine
from flask_oauthlib.client import OAuth
from flask_pymongo import PyMongo
from raven.contrib.flask import Sentry
from werkzeug.middleware.profiler import ProfilerMiddleware

from jafar.storages.memmap import MemmapStorageWrapper
from version import VERSION
from jafar.app_info_loader import AppInfoLoader, VangaGeneralStatsLoader, TopClassLoader
from extensions.s3mds import S3Client
from extensions.resizer import Resizer

__version__ = VERSION

db = MongoEngine()
admin_app = Admin(template_mode='bootstrap3', base_template='base.html')
cache = Cache()
shared_cache = Cache(config={'CACHE_TYPE': 'redis'})
fast_cache = Cache(config={'CACHE_TYPE': 'jafar.utils.cache.fast_simple'})
request_cache = Cache(config={'CACHE_TYPE': 'jafar.utils.cache.request_cache'})
sentry = Sentry()
advisor_mongo = PyMongo()
advisor_replica_mongo = PyMongo()
localization_mongo = PyMongo()
jafar_mongo = PyMongo()
storage_wrapper = MemmapStorageWrapper()
login_manager = LoginManager()
oauth = OAuth()
s3_client = S3Client()
app_info_loader = AppInfoLoader()
vanga_general_stats_loader = VangaGeneralStatsLoader()
top_class_name_loader = TopClassLoader()
clickhouse = ClickHouse()
resizer = Resizer()

yandex_oauth = oauth.remote_app(
    'yandex-team', app_key='OAUTH_YANDEX'
)


@login_manager.user_loader
def load_user(email):
    from jafar.admin.models.base import AdminUser

    return AdminUser.objects(email=email).first()


def _db_init(app, replica_data):
    if not replica_data:
        advisor_mongo.init_app(app, config_prefix='ADVISOR_MONGO')
        advisor_replica_mongo.init_app(app, config_prefix='ADVISOR_MONGO_REPLICA')
    else:
        advisor_mongo.init_app(app, config_prefix='ADVISOR_MONGO_REPLICA')
    localization_mongo.init_app(app, config_prefix='LOCALIZATION_MONGO')
    jafar_mongo.init_app(app)
    clickhouse.init_app(app)

    # configuration for unittests
    # a little bit black magic, most of which is explained in ADVISOR-791
    if app.config['TESTING']:
        # prevent flask-mongoengine from connecting to real database
        with patch('flask_mongoengine.connection._connect'):
            db.init_app(app)
        with app.app_context():
            # mock mongoengine connect
            jafar_connection = mongoengine.connect(app.config['MONGO_DBNAME'], host='mongomock://localhost')
            advisor_connection = mongoengine.connect(app.config['ADVISOR_MONGO_DBNAME'], host='mongomock://localhost',
                                                     alias='advisor')

            # flask-mongoengine keeps a separate connection object in app.extensions
            app.extensions['mongoengine'][db]['conn'] = {'default': jafar_connection, 'advisor': advisor_connection}
            # pymongo connects
            for prefix in ('ADVISOR_MONGO', 'ADVISOR_MONGO_REPLICA'):
                client = mongomock.MongoClient()
                app.extensions['pymongo'][prefix] = (client, client.db)
    else:
        if not replica_data:
            db.init_app(app)
        else:
            db.init_app(app, config={'MONGODB_SETTINGS': app.config['MONGODB_REPLICA_SETTINGS']})
        jafar_connection = mongoengine.connection.get_connection()
        advisor_connection = mongoengine.connection.get_connection('advisor')

    with app.app_context():
        # force advisor_mongo and jafar_mongo to use the same connection instance as mongoengine
        app.extensions['pymongo'][advisor_mongo.config_prefix] = (
            advisor_connection, advisor_connection.get_database(name=app.config['ADVISOR_MONGO_DBNAME'])
        )
        app.extensions['pymongo'][jafar_mongo.config_prefix] = (
            jafar_connection, jafar_connection.get_database(name=app.config['MONGO_DBNAME'])
        )


    # ensure that all mongo indexes exist
    if not app.config['TESTING'] and not app.config['READ_ONLY']:
        from jafar.commands.ensure_indexes import EnsureIndexes

        with app.app_context():
            EnsureIndexes().run()


def create_app(profile=False, profile_dir=None, disable_logging=False,
               storage_mode=None, replica_data=False, **config_overrides):
    # intitializing app
    app = Flask(__name__)
    app.config.from_object('settings')
    app.config.update(config_overrides)
    if storage_mode:
        app.config.update(MEMMAP_STORAGE_MODE=storage_mode)

    if replica_data:
        # you can't write to replica
        app.config['READ_ONLY'] = True

    # registering views
    from jafar.views import common_blueprint
    from jafar.feed.urls import feed_blueprint
    from jafar.data_providers.launcher.blacklist import blacklist_caches

    app.register_blueprint(common_blueprint)
    app.register_blueprint(feed_blueprint)

    # register admin views
    from jafar.admin.views import JafarAdminIndexView, OAuthCallback
    import jafar.admin.urls

    app.route('/oauth_callback')(OAuthCallback.as_view('oauth_callback'))

    # enable disabled logging back
    logging.disable(logging.NOTSET)

    # init third-party services
    admin_app.init_app(app, index_view=JafarAdminIndexView())
    admin_app.name = '%s v%s' % (app.config['ADMIN_LOGO'], VERSION)
    cache.init_app(app)
    shared_cache.init_app(app, config={
        'CACHE_REDIS_HOST': app.config['SHARED_CACHE_REDIS_HOST'],
        'CACHE_REDIS_PORT': app.config['SHARED_CACHE_REDIS_PORT'],
        'CACHE_REDIS_PASSWORD': app.config['SHARED_CACHE_REDIS_PASSWORD'],
        'CACHE_REDIS_DB': app.config['SHARED_CACHE_REDIS_DB'],
    })
    fast_cache.init_app(app)
    request_cache.init_app(app)
    app_info_loader.init_app(app)
    vanga_general_stats_loader.init_app(app)
    top_class_name_loader.init_app(app)

    sentry.init_app(app, dsn=app.config['SENTRY_DSN'], level=logging.WARNING)
    sentry.client.release = VERSION
    sentry.client.environment = yenv.type

    # logging config
    if not disable_logging:
        logging_config = app.config['LOGGING']
        logging_config['handlers']['sentry']['client'] = sentry.client
        logging.config.dictConfig(logging_config)

    Bootstrap(app)
    storage_wrapper.init_app(app)
    login_manager.init_app(app)

    _db_init(app, replica_data)

    if profile:
        app.wsgi_app = ProfilerMiddleware(app.wsgi_app, profile_dir=profile_dir)

    with app.app_context():
        for blacklist_cache in blacklist_caches.values():
            blacklist_cache.force_reload()

    s3_client.init_app(app)
    resizer.init_app(app)
    return app
