import logging

import iso3166
import requests
from flask import current_app as app
from flask_admin.model.ajax import AjaxModelLoader, DEFAULT_PAGE_SIZE

from jafar import localization_mongo
from jafar.models.yandexphone import Region

MIN_PACKAGE_NAME_LENGTH = 5
MIN_PUBLISHER_NAME_LENGTH = 3

logger = logging.getLogger(__name__)


class StringAjaxLoader(AjaxModelLoader):
    def __init__(self, name, **options):
        super(StringAjaxLoader, self).__init__(name, options)

    def format(self, model):
        """ model is just a string, so return it as id and as name to show in form """
        if not model:
            return None
        return model, model

    def get_one(self, pk):
        """ String id is a string itself """
        return pk


class TranslationsAjaxLoader(StringAjaxLoader):
    @property
    def translations(self):
        return localization_mongo.db[app.config['LAUNCHER_TRANSLATIONS_COLLECTION']]

    def format(self, model):
        """ model is just a string, so return it as id and as name to show in form """
        if not model:
            return None
        translation = self.translations.find_one({'_id': model})
        ru_value = u''
        if translation:
            for value in translation['values']:
                if value['conditions'].get('locale', {}).get('language') == 'ru':
                    ru_value = value['value']
                    break
        return model, u'%s - %s' % (model, ru_value)

    def get_list(self, query, offset=0, limit=DEFAULT_PAGE_SIZE):
        """ Load possible translation keys from localization mongo """
        query = {'_id': {'$regex': query}}
        return [doc['_id'] for doc in self.translations.find(query, skip=offset, limit=limit)]

    def get_translation_list(self, key):
        """ Get strings in all locales by the key """
        translation = self.translations.find_one({'_id': key})
        if not translation:
            logger.debug('No translation found for key %s' % key)
            return None
        result = {}
        for item in translation['values']:
            result[item['conditions']['locale']['language']] = item['value']
        return result


class CountryAjaxLoader(StringAjaxLoader):
    countries = sorted(country.alpha2 for country in iso3166.countries)

    def get_list(self, query, offset=0, limit=DEFAULT_PAGE_SIZE):
        query = query.upper()
        filtered_countries = filter(lambda country: country.startswith(query), self.countries)
        return filtered_countries[offset:offset + limit]


class GeoCodesAjaxModelLoader(AjaxModelLoader):
    delimiter = '_'

    def __init__(self, name, **options):
        super(GeoCodesAjaxModelLoader, self).__init__(name, options)

    def get_pk(self, code, description):
        # NOTE: hex-encoding is required because flask interptets comma-separated
        # strings as multiple form values
        return u"{code}{delimiter}{description}".format(
            code=code, delimiter=self.delimiter, description=description
        ).encode('utf8').encode('hex')

    def format(self, model):
        """ formats model object
        :param model: can be a DBRef instance or tuple
        :return: tuple(id, value) or None
        """
        if not model:
            return None
        if isinstance(model, tuple):
            return model
        return self.get_pk(model.code, model.description), model.description

    def get_one(self, pk):
        code, description = pk.decode('hex').decode('utf8').split(self.delimiter)
        return Region(code, description)

    def get_list(self, query, offset=0, limit=DEFAULT_PAGE_SIZE):
        result = []
        for suggest in self.lookup_region_info(query):
            code = suggest['geoid']
            description = '%s (%s)' % (suggest['name_short'], suggest['name'])
            result.append((self.get_pk(code, description), description))
        return result

    @classmethod
    def lookup_region_info(cls, name):
        """
        :param name: substring of region name
        :return: array of region definitions. see module's docstring
        """
        logger.debug(u'lookup_region_info: name=%r', name)

        response = requests.get(
            app.config['GEO_SUGGEST_URL'],
            params={
                'part': name,
                'type': 'addr',
                'v': 8,  # this response format includes geoid,
                'bases': 'district,locality,area,province,country',  # lookup only regions from district to country,
                'callback': '',  # don't include JSONP callback in response,
            }
        )
        response.raise_for_status()
        _, response_body = response.json()
        for suggest in response_body:
            if 'geoid' not in suggest:
                continue
            yield suggest
