import json
from datetime import datetime

import pytz
import six
from flask import current_app as app
from flask_admin.form import FileUploadField
from flask_mongoengine.wtf.fields import JSONField
from wtforms import StringField
from wtforms.fields.html5 import DateTimeField

from jafar import s3_client
from jafar.admin.mds import LauncherMdsClient
from jafar.admin.models.wallpapers import get_image_metadata
from jafar.admin.widgets import ColorWidget, CompositeCoverWidget, S3ImageUploadInput

launcher_mds_client = LauncherMdsClient()


class PrettyJSONField(JSONField):
    """
    Like a JSONField, but uses indent to prettify the output and doesn't escape unicode symbols
    """

    def _value(self):
        if self.raw_data:
            return self.raw_data[0]
        else:
            return json.dumps(self.data, indent=4, encoding='utf-8', ensure_ascii=False) if self.data else u''


class BaseLocalDateTimeField(DateTimeField):
    def process_data(self, value):
        super(BaseLocalDateTimeField, self).process_data(value)
        try:
            # converting datetime from UTC to MSK timezone for representation in form
            self.data = self.data.replace(tzinfo=pytz.UTC).astimezone(app.config['LOCAL_TIMEZONE'])
        except AttributeError:
            pass


class LocalDateTimeField(BaseLocalDateTimeField):
    def process_formdata(self, valuelist):
        super(LocalDateTimeField, self).process_formdata(valuelist)
        # If get datetime with time greater then 00:00:00, interpret it as 23:59:59:99999 (max time)
        # Doing so makes 'end_date' date to be included
        if self.data.time() > datetime.min.time():
            self.data = datetime.combine(self.data.date(), datetime.max.time())

        # converting inputed datetime from MSK to UTC timezone before saving to database
        # all calculations are made with UTC timezone
        self.data = app.config['LOCAL_TIMEZONE'].localize(self.data, is_dst=None).astimezone(pytz.UTC)


class MdsFileUploadField(FileUploadField):
    def __init__(self, kind, **kwargs):
        self.kind = kind
        super(MdsFileUploadField, self).__init__(allow_overwrite=True, **kwargs)

    def _get_path(self, filename):
        return filename

    def _delete_file(self, mds_key):
        launcher_mds_client.delete(mds_key)

    def _save_file(self, data, filename):
        mds_key = launcher_mds_client.save(filename=filename, data=data.read(), kind=self.kind)
        return mds_key


class S3FileUploadField(FileUploadField):
    def _delete_file(self, mds_key):
        pass

    def _get_path(self, filename):
        return filename

    # noinspection PyUnusedLocal,PyMethodMayBeStatic
    def get_metadata(self, data, filename):
        """ Override to write additional metadata to MDS """
        return {'size': data.content_length}

    def _save_file(self, data, filename):
        metadata = self.get_metadata(data, filename)
        s3_key = s3_client.save(key_name=filename, data=data.read(), metadata=metadata, replace=False)
        return s3_key.name

    def populate_obj(self, obj, name):
        """ Use existing if file is already uploaded (model copy case) """
        if isinstance(self.data, six.string_types):
            setattr(obj, name, self.data)
        else:
            super(S3FileUploadField, self).populate_obj(obj, name)


class ColorField(StringField):
    """
    Represents an ``<input type="color">``.
    This allows showing and picking any color.
    """
    widget = ColorWidget()


class S3ImageUploadField(S3FileUploadField):
    widget = S3ImageUploadInput()

    def get_metadata(self, data, filename):
        metadata = super(S3ImageUploadField, self).get_metadata(data, filename)
        metadata.update(get_image_metadata(data))
        data.stream.seek(0)
        return metadata

    def _save_file(self, data, filename):
        metadata = self.get_metadata(data, filename)
        s3_key = s3_client.save(key_name=filename, data=data.read(), metadata=metadata, replace=False)
        return s3_key.name


class CompositeCoverField(S3ImageUploadField):
    widget = CompositeCoverWidget()
