from flask_admin import form
from flask_admin.contrib.mongoengine import EmbeddedForm
from flask_admin.contrib.mongoengine.fields import ModelFormField, MongoFileField, MongoImageField
from flask_admin.contrib.mongoengine.form import get_form
from flask_admin.model.fields import InlineFieldList, AjaxSelectField
from flask_admin.model.form import FieldPlaceholder
from flask_mongoengine.wtf import orm
from mongoengine import ListField
from wtforms import validators, fields


class CustomModelConverter(orm.ModelConverter):
    """
        Customized MongoEngine form conversion class.
        Injects various Flask-Admin widgets and handles lists with
        customized InlineFieldList field.
    """

    def __init__(self, view):
        super(CustomModelConverter, self).__init__()

        self.view = view

    def _get_field_override(self, name):
        form_overrides = getattr(self.view, 'form_overrides', None)

        if form_overrides:
            return form_overrides.get(name)

        return None

    def _get_subdocument_config(self, name):
        config = getattr(self.view, '_form_subdocuments', {})

        p = config.get(name)
        if not p:
            return EmbeddedForm()

        return p

    def _convert_choices(self, choices):
        for c in choices:
            if isinstance(c, tuple):
                yield c
            else:
                yield (c, c)

    def clone_converter(self, view):
        return self.__class__(view)

    def convert(self, model, field, field_args):
        # Check if it is overridden field
        if isinstance(field, FieldPlaceholder):
            return form.recreate_field(field.field)

        kwargs = {
            'label': getattr(field, 'verbose_name', None),
            'description': getattr(field, 'help_text', ''),
            'validators': [],
            'filters': [],
            'default': field.default
        }

        if field_args:
            kwargs.update(field_args)

        if kwargs['validators']:
            # Create a copy of the list since we will be modifying it.
            kwargs['validators'] = list(kwargs['validators'])

        if field.required:
            kwargs['validators'].append(validators.InputRequired())
        elif not isinstance(field, ListField):
            kwargs['validators'].append(validators.Optional())

        ftype = type(field).__name__

        if field.choices:
            kwargs['choices'] = list(self._convert_choices(field.choices))

            if ftype in self.converters:
                kwargs["coerce"] = self.coerce(ftype)
            if kwargs.pop('multiple', False):
                return fields.SelectMultipleField(**kwargs)
            return fields.SelectField(**kwargs)

        if hasattr(field, 'to_form_field'):
            return field.to_form_field(model, kwargs)

        override = self._get_field_override(field.name)
        if override:
            return override(**kwargs)

        if ftype in self.converters:
            return self.converters[ftype](model, field, kwargs)

    @orm.converts('DateTimeField')
    def conv_DateTime(self, model, field, kwargs):
        kwargs['widget'] = form.DateTimePickerWidget()
        return orm.ModelConverter.conv_DateTime(self, model, field, kwargs)

    @orm.converts('ListField')
    def conv_List(self, model, field, kwargs):
        if field.field is None:
            raise ValueError('ListField "%s" must have field specified for model %s' % (field.name, model))

        # Disabling multi select because of problems with sorting references. Might become usefull later
        """
        if isinstance(field.field, ReferenceField):
            loader = getattr(self.view, '_form_ajax_refs', {}).get(field.name)
            if loader:
                return AjaxSelectMultipleField(loader, **kwargs)

            kwargs['widget'] = form.Select2Widget(multiple=True)

            # TODO: Support AJAX multi-select (This is TODO from library)
            doc_type = field.field.document_type
            return mongo_fields.ModelSelectMultipleField(model=doc_type, **kwargs)
        """

        # Create converter
        view = self._get_subdocument_config(field.name)
        converter = self.clone_converter(view)

        if field.field.choices:
            kwargs['multiple'] = True
            return converter.convert(model, field.field, kwargs)

        unbound_field = converter.convert(model, field.field, {})
        min_entries = int(field.required)
        return InlineFieldList(unbound_field, min_entries=min_entries, **kwargs)

    @orm.converts('EmbeddedDocumentField')
    def conv_EmbeddedDocument(self, model, field, kwargs):
        # FormField does not support validators
        kwargs['validators'] = []

        view = self._get_subdocument_config(field.name)

        form_opts = form.FormOpts(widget_args=getattr(view, 'form_widget_args', None),
                                  form_rules=view._form_rules)

        form_class = view.get_form()
        if form_class is None:
            converter = self.clone_converter(view)
            form_class = get_form(field.document_type_obj, converter,
                                  base_class=view.form_base_class or form.BaseForm,
                                  only=view.form_columns,
                                  exclude=view.form_excluded_columns,
                                  field_args=view.form_args,
                                  extra_fields=view.form_extra_fields)

            form_class = view.postprocess_form(form_class)

        return ModelFormField(field.document_type_obj, view, form_class, form_opts=form_opts, **kwargs)

    @orm.converts('ReferenceField')
    def conv_Reference(self, model, field, kwargs):
        kwargs['allow_blank'] = not field.required

        loader = getattr(self.view, '_form_ajax_refs', {}).get(field.name)
        if loader:
            return AjaxSelectField(loader, **kwargs)

        kwargs['widget'] = form.Select2Widget()

        return orm.ModelConverter.conv_Reference(self, model, field, kwargs)

    @orm.converts('FileField')
    def conv_File(self, model, field, kwargs):
        return MongoFileField(**kwargs)

    @orm.converts('ImageField')
    def conv_image(self, model, field, kwargs):
        return MongoImageField(**kwargs)

