from datetime import datetime

from flask import current_app as app
from flask_login import UserMixin
from mongoengine import EmbeddedDocument, StringField, IntField, Document, DateTimeField, ListField, DictField

from jafar import s3_client

ROLE_ADMIN = 'admin'
ROLE_DEVELOPER = 'developer'
ROLE_AD_MANAGER = 'ad_manager'
ROLE_TESTER = 'tester'
ROLE_RESOURCE = 'resource_manager'
ROLE_SUPPORT = 'support'
ROLE_ASSESSOR = 'assessor'
ROLES = (ROLE_ADMIN, ROLE_DEVELOPER, ROLE_AD_MANAGER, ROLE_TESTER, ROLE_RESOURCE, ROLE_SUPPORT, ROLE_ASSESSOR)


class S3Image(EmbeddedDocument):
    key = StringField(required=False, default=None)
    width = IntField(required=False, default=0)
    height = IntField(required=False, default=0)
    hash = StringField(required=False)

    meta = {
        'strict': False,
    }

    def validate(self, clean=True):
        key = s3_client.get_key(self.key)
        if key:
            self.width = key.metadata['width']
            self.height = key.metadata['height']
            self.hash = key.etag.strip('"')
        return super(S3Image, self).validate(clean)

    @property
    def url(self):
        return app.config['S3_URL_TEMPLATE'] % self.key

    def __unicode__(self):
        return self.key

    def get_data(self):
        key = s3_client.get_key(self.key)
        return key.read()

    def exists(self):
        key = s3_client.get_key(self.key)
        return key is not None and key.exists()


class LauncherFiles(Document):
    md5 = StringField(required=True)
    filename = StringField(required=True)
    mds_key = StringField(db_field='new_mds_filename', required=True)
    kind = StringField(required=True)
    updated_at = DateTimeField(required=False, default=datetime.utcnow)

    meta = {
        'db_alias': 'advisor',
        'strict': False
    }


class YandexCategory(Document):
    name = StringField(required=True, db_field='yandex_name')
    google_names = ListField(field=StringField(), required=False, db_field="google_name")
    title = DictField(required=True, db_field='yandex_title')

    meta = {
        'db_alias': 'advisor',
        'collection': 'categories',
        'strict': False,
    }


class History(Document):
    user = StringField()
    action = StringField()
    document_class = StringField()
    document_before = StringField(required=False)
    document_after = StringField(required=False)
    timestamp = DateTimeField(default=datetime.utcnow)

    meta = {
        'strict': False,
    }


class AdminUser(Document, UserMixin):
    email = StringField(max_length=255)
    username = StringField(max_length=255)
    roles = ListField(StringField(choices=ROLES), default=[])

    def has_role(self, role):
        return role in self.roles

    def get_id(self):
        return self.email

    def __unicode__(self):
        return self.username

    meta = {
        'strict': False,
        'indexes': ['email'],
        'auto_create_index': False,
        'index_background': True,
    }
