from collections import OrderedDict
from datetime import datetime

from mongoengine import (
    Document, ValidationError,
    StringField, ListField,
    DateTimeField, IntField,
    EmbeddedDocumentField
)

from jafar.admin.models.base import S3Image


class Icons(Document):
    DPI = OrderedDict((
        ('mdpi', 48),
        ('hdpi', 72),
        ('xhdpi', 96),
        ('xxhdpi', 144),
        ('xxxhdpi', 192),
    ))
    PACKS = ('pillow', 'circle')

    package_name = StringField(required=True)
    activity = StringField(required=False, null=True)
    language = StringField(required=False, null=True, regex=r'^([a-z]{2})?$')
    pack = StringField(required=True, choices=PACKS)
    clid1 = ListField(StringField(regex=r'^\d+$'), required=False, null=True)
    mdpi = EmbeddedDocumentField(S3Image, required=True)
    hdpi = EmbeddedDocumentField(S3Image, required=True)
    xhdpi = EmbeddedDocumentField(S3Image, required=True)
    xxhdpi = EmbeddedDocumentField(S3Image, required=True)
    xxxhdpi = EmbeddedDocumentField(S3Image, required=True)

    updated_at = DateTimeField(required=False, default=datetime.utcnow)

    meta = {
        'db_alias': 'advisor',
        'strict': False,
        'indexes': ['package_name'],
    }

    def clean(self):
        if not self.clid1:
            self.clid1 = None
        if not self.language:
            self.language = None
        if not self.activity:
            self.activity = None

    def validate(self, *args, **kwargs):
        super(Icons, self).validate(*args, **kwargs)
        for dpi, size in self.DPI.items():
            image = self[dpi]
            for dimension in ('width', 'height'):
                actual_size = int(image[dimension])
                if actual_size != size:
                    raise ValidationError(
                        'Icon {dimension} for {dpi} is equal {actual_size}, not {size}'.format(
                            dimension=dimension, dpi=dpi, size=size, actual_size=actual_size,
                        ))
