import base64
import io
import logging
import hashlib
from datetime import datetime
from zipfile import ZipFile

import requests
import six
from PIL import Image, ImageOps
from flask import flash
from mongoengine import (
    Document, EmbeddedDocument,
    StringField, IntField,
    BooleanField, ListField,
    DateTimeField, EmbeddedDocumentField,
    ReferenceField
)

from jafar import s3_client
from jafar.admin.models.base import S3Image
from jafar.image_composer import ImageComposer, XXXHDPI_DENSITY
from jafar.admin.ajax import TranslationsAjaxLoader
from jafar.utils import safe_get, to_ascii

API_YA_CC_TIMEOUT = 1

# TODO: move these to base or some other place
BASE64_PREVIEW_QUALITY = 90
BASE64_PREVIEW_FORMAT = 'JPEG'
BASE64_PREVIEW_SIZE = (32, 32)
BASE64_PREVIEW_MODE = 'RGB'

COLOR_REGEX = r'#[0-9A-Fa-f]{8}|#[0-9A-Fa-f]{6}'
COLOR_BLACK = '#000000'
LANGUAGE_CHOICES = ('en', 'es', 'pt', 'ru', 'be', 'kk', 'tr', 'uk')

# This key is used for image, that must be used for all locales
# This means, that text is not provided, for cover, and we can use 1 cover for all locales
KEY_SINGLE_IMAGE = 'single_image'

logger = logging.getLogger(__name__)


def make_short_url(url):
    response = requests.get('http://api.ya.cc/--', params={'url': url}, timeout=API_YA_CC_TIMEOUT)
    response.raise_for_status()
    return response.text


def expand_short_url(url_short):
    response = requests.head(url_short, timeout=API_YA_CC_TIMEOUT, allow_redirects=False)
    return response.headers['Location']


def get_image_metadata(data):
    """TODO: move to fields/base/somewhere else"""
    if isinstance(data, six.string_types):
        data = io.BytesIO(data)
    image = Image.open(data)
    return {
        'width': image.width,
        'height': image.height,
        'mimetype': Image.MIME[image.format],
    }


class WallpaperCovers(EmbeddedDocument):
    DPI = ('full', 'xxxhdpi', 'xxhdpi', 'xhdpi', 'hdpi', 'mdpi')

    language = StringField(required=True, choices=LANGUAGE_CHOICES)
    image_full = EmbeddedDocumentField(S3Image, required=True, default=S3Image)

    meta = {
        'strict': False,
    }


class WallpaperColors(Document):
    group = StringField(required=True)
    id_ = StringField(required=True)
    value = StringField(required=True)
    order = IntField(required=True)

    def __unicode__(self):
        return u'{}/{}'.format(self.group, self.id_)

    meta = {
        'db_alias': 'advisor',
        'indexes': ['group', 'id_'],
        'strict': False,
    }


class WallpaperBadges(Document):
    id_ = StringField(required=True, primary_key=True)
    text_color = StringField(required=True)
    background_color = StringField(required=True)

    def __unicode__(self):
        return u'{}'.format(self.id_)

    meta = {
        'db_alias': 'advisor',
        'strict': False,
    }


class WallpaperImages(EmbeddedDocument):
    priority = IntField(required=False, default=0)
    image = EmbeddedDocumentField(S3Image, required=False, default=S3Image)
    url_short = StringField(required=False)
    preview_base64 = StringField(required=False)
    colors = ListField(ReferenceField(WallpaperColors), required=False)
    badges = ListField(ReferenceField(WallpaperBadges), required=False)

    meta = {
        'strict': False,
    }

    def clean(self):
        self.image.clean()

        if self._get_changed_fields():
            self.preview_base64 = self.make_base_64_preview(self.image.get_data())

        url = self.image.url
        if not self.url_short or expand_short_url(self.url_short) != url:
            self.url_short = make_short_url(url)

    @staticmethod
    def make_base_64_preview(image_data):
        full_image = Image.open(io.BytesIO(image_data))
        if full_image.mode != BASE64_PREVIEW_MODE:
            full_image = full_image.convert(BASE64_PREVIEW_MODE)
        cropped_image = ImageOps.fit(full_image, BASE64_PREVIEW_SIZE)
        buffered = io.BytesIO()
        cropped_image.save(buffered, format=BASE64_PREVIEW_FORMAT, quality=BASE64_PREVIEW_QUALITY)
        return base64.b64encode(buffered.getvalue())


class ColoredText(EmbeddedDocument):
    text = StringField(required=False)
    color = StringField(required=False, regex=COLOR_REGEX, default=COLOR_BLACK)

    meta = {
        'strict': False,
    }


class CompositeCoverBackground(EmbeddedDocument):
    image = EmbeddedDocumentField(S3Image, required=False)
    color = StringField(required=False, regex=COLOR_REGEX, default=COLOR_BLACK)
    has_gradient = BooleanField(required=False)

    meta = {
        'strict': False,
    }


class CompositeCover(EmbeddedDocument):
    title = EmbeddedDocumentField(ColoredText, required=True)
    subtitle = EmbeddedDocumentField(ColoredText, required=False)
    background = EmbeddedDocumentField(CompositeCoverBackground, required=False)
    full_size = BooleanField(required=False, default=False)
    generated_covers = ListField(EmbeddedDocumentField(WallpaperCovers), required=False)

    meta = {
        'strict': False,
    }


class WallpaperCategories(Document):
    COLLECTION_API = 'wallpaper_categories'
    COLLECTION_EDITABLE = 'wallpaper_categories_editable'
    COLLECTION_TEMP = 'wallpaper_categories_editable_temp'
    COLLECTION_BACKUP = 'wallpaper_categories_backup'

    metadata_file_name = 'metadata.json'

    name = StringField(required=True, unique=True)
    covers = ListField(EmbeddedDocumentField(WallpaperCovers), required=False)
    composite_cover = EmbeddedDocumentField(CompositeCover, required=False)
    images = ListField(EmbeddedDocumentField(WallpaperImages), required=False)
    use_in_autochange = BooleanField(required=False, default=True)
    use_in_feed = BooleanField(required=False, default=True)
    updated_at = DateTimeField(required=False)

    meta = {
        'db_alias': 'advisor_primary',
        'strict': False,
        'collection': COLLECTION_EDITABLE,
    }

    def get_cache_break(self):
        """
        Provides unique value(since it unique in any time moment and very lightweight)

        Used to modify the URL of image, so it will be loaded from source(not cache) every time
        :return: timestamp as string
        """
        return str(int(datetime.now().microsecond))

    def save(self, *args, **kwargs):
        self.images = sorted(self.images, key=lambda x: -x.priority)
        self.updated_at = datetime.utcnow()
        self._save_generated_covers_to_s3(self._generate_previews_for_composite_cover())
        return super(WallpaperCategories, self).save(*args, **kwargs)

    def __unicode__(self):
        return u'name: %s, covers: %d, images: %d' % (
            self.name, len(self.covers), len(self.images),
        )

    def export(self):
        buf = io.BytesIO()
        with ZipFile(buf, mode='w') as zip_file:
            del self.id
            zip_file.writestr(self.metadata_file_name, self.to_json())
            for image in self.all_images:
                data = image.get_data()
                zip_file.writestr(image.key, data)
        buf.seek(0)
        return buf

    @classmethod
    def import_(cls, file_object, replace):
        with ZipFile(file_object, mode='r') as zip_file:
            metadata = zip_file.read(cls.metadata_file_name)
            model = cls.from_json(metadata)
            existing_objects = cls.objects(name=model.name)
            if existing_objects.count() > 0:
                if not replace:
                    flash('Category with name "%s" already exists' % model.name, 'error')
                    return
                else:
                    existing_objects.delete()
            for image in model.all_images:
                if not image.exists():
                    image_data = zip_file.read(image.key)
                    metadata = get_image_metadata(image_data)
                    s3_client.save(image_data, image.key, metadata)
                    logger.info('Saved new image to S3: %s', image.key)
            model.save(force_insert=True)

    @property
    def all_images(self):
        for cover in self.covers:
            yield cover.image_full
        for wallpaper in self.images:
            yield wallpaper.image
        if self.composite_cover.background.image.key:
            yield self.composite_cover.background.image

    @classmethod
    def copy_collection(cls, source, destination):
        """
        use this method to make hard copy during reset operation or
        make automatically copy when beginning of editing process
        """
        db = cls._get_db()
        db[cls.COLLECTION_TEMP].drop()
        # copy all docs to temporary collection
        db[source].aggregate([{'$match': {}}, {'$out': cls.COLLECTION_TEMP}])
        # just rename collection as transactional operation
        db[cls.COLLECTION_TEMP].rename(destination, dropTarget=True)

    @classmethod
    def backup(cls):
        cls.copy_collection(cls.COLLECTION_API, cls.COLLECTION_BACKUP)

    @classmethod
    def restore(cls):
        cls.copy_collection(cls.COLLECTION_BACKUP, cls.COLLECTION_API)

    @classmethod
    def reset(cls):
        cls.copy_collection(cls.COLLECTION_API, cls.COLLECTION_EDITABLE)

    @classmethod
    def apply(cls):
        cls.copy_collection(cls.COLLECTION_EDITABLE, cls.COLLECTION_API)

    def _merge_titles_and_subtitles(self, titles, subtitles):
        """
        Merges 2 dictionaries with localized titles and subtitles into one dictionary
        For example:
        titles: {'ru': 'Title 1'<in russian language>, 'en': 'Title 1', ...}
        subtitles: {'en': 'Subtitle 1', 'ru': 'Subtitle 1'<In russian language>}

        The result will be:
        {
            {'en': {'title': 'Title 1', 'subtitle': 'Subtitle 1'}},
            {'ru': {'title': 'Title 1', 'subtitle': 'Subtitle 1'}},<In russian language>
            ...
        }

        If localization for certain country code not found - this country code will be skipped from the result

        :param titles: dictionary with locale as keys, and corresponding text in this locale as value
        :param subtitles: dictionary with locale as keys, and corresponding text in this locale as value
        :return: dictionary, with locale as a key, and object with title and subtitle as value
        """
        result = {}
        for key in LANGUAGE_CHOICES:
            item = {}
            if titles and key in titles:
                item['title'] = titles[key]
            if subtitles and key in subtitles:
                item['subtitle'] = subtitles[key]
            if 'title' in item or 'subtitle' in item:
                result[key] = item
        return result

    def _generate_previews_for_composite_cover(self):
        # obtain localized strings for title and subtitle
        loader = TranslationsAjaxLoader('composite_cover-title-text')
        title_key = safe_get(self, 'composite_cover', 'title', 'text')
        subtitle_key = safe_get(self, 'composite_cover', 'subtitle', 'text')

        # build dict with country code as key, and title + subtitle as value
        composite_text = self._merge_titles_and_subtitles(loader.get_translation_list(title_key),
                                                          loader.get_translation_list(subtitle_key))

        # Cache image building params to use in 'for loop'
        image = safe_get(self, 'composite_cover', 'background', 'image')
        background_color = to_ascii(safe_get(self, 'composite_cover', 'background', 'color'))
        has_gradient = safe_get(self, 'composite_cover', 'background', 'has_gradient')
        full_size = safe_get(self, 'composite_cover', 'full_size')
        title_text_color = to_ascii(safe_get(self, 'composite_cover', 'title', 'color'))
        subtitle_text_color = to_ascii(safe_get(self, 'composite_cover', 'subtitle', 'color'))

        generated_image_list = {}
        if image and image.exists() and background_color:
            image_data = io.BytesIO(image.get_data())
            #  If text specified - generate images without text
            if len(composite_text) == 0:
                composer = ImageComposer(image_data, background_color, XXXHDPI_DENSITY)
                composer.set_full_size(full_size)
                if has_gradient:
                    composer.set_full_size(background_color)
                generated_image_list[KEY_SINGLE_IMAGE] = composer.compose()
            else:
                for key in LANGUAGE_CHOICES:
                    if key in composite_text.keys():  # skip locale without text
                        composer = ImageComposer(image_data, background_color, XXXHDPI_DENSITY)
                        title = safe_get(composite_text, key, 'title')
                        if title and title_text_color:
                            composer.set_title(title, title_text_color)
                        subtitle = safe_get(composite_text, key, 'subtitle')
                        if subtitle and subtitle_text_color:
                            composer.set_subtitle(subtitle, subtitle_text_color)
                        if has_gradient:
                            composer.add_gradient(background_color)
                        if full_size:
                            composer.set_full_size(full_size)

                        generated_image_list[key] = composer.compose()
        else:
            logger.debug("Expected image and background color to be not None")
        return generated_image_list

    def _save_generated_covers_to_s3(self, generated_images):
        result = {}
        if KEY_SINGLE_IMAGE in generated_images:
            # upload on s3 only 1 image for all locales
            keys = [KEY_SINGLE_IMAGE]  # Suffix, describes that image is for all locales
        else:
            # upload different images for locales
            keys = LANGUAGE_CHOICES
        for key in keys:
            # Save the image binary data to bytes
            in_mem_file = io.BytesIO()
            image = generated_images.get(key, None)
            # if no image for certain locale - skip it
            if image is None:
                continue
            image = image.convert('RGB')
            image.save(in_mem_file,
                       format='JPEG',
                       quality=100,
                       optimize=True,
                       progressive=True)
            meta = {'size': image.width * image.height,
                    'width': image.width,
                    'height': image.height, }

            # Save image on S3 storage
            s3_key = s3_client.save(key_name='cover_{}_{}.jpg'.format(self.name, key),
                                    data=in_mem_file.getvalue(),
                                    metadata=meta,
                                    replace=True)
            md5 = hashlib.md5('size={},width={},height={}'.format(meta['size'], meta['width'], meta['height']))\
                .hexdigest()
            s3_image = S3Image(key=s3_key.key, width=image.width, height=image.height, hash=md5)

            # Put uploaded image to the collection, in order to put it to model later
            result.update({key: s3_image})

        # detect if only one image must be used for all locales
        use_one_image_for_all = KEY_SINGLE_IMAGE in result and len(result.keys()) == 1
        covers = []
        for locale in LANGUAGE_CHOICES:
            if locale not in result:
                continue  # ignore if there is no image for such locale
            covers.append(WallpaperCovers(
                language=locale,
                image_full=result[KEY_SINGLE_IMAGE if use_one_image_for_all else locale]))
        # put covers to model
        self.composite_cover.generated_covers = covers


class MusicWallpapers(Document):
    name = StringField(required=False)
    image = EmbeddedDocumentField(S3Image, required=False, default=S3Image)
    updated_at = DateTimeField(required=False)

    meta = {
        'db_alias': 'advisor',
        'collection': 'music_wallpapers',
        'strict': False,
    }

    def save(self, *args, **kwargs):
        self.updated_at = datetime.utcnow()
        return super(MusicWallpapers, self).save(*args, **kwargs)
