import re
import logging
from datetime import datetime

import requests
from flask import current_app as app

from jafar import localization_mongo

logger = logging.getLogger('stream')

re_strip = re.compile(ur'^[\u202b\u202c]*(.*?)[\u202b\u202c]*$', re.U)

TANKER_FAVICON = 'https://tanker.yandex-team.ru/static/i/favicon.ico'
TANKER_SHOW_URL = 'http://tanker.yandex-team.ru/?project={project}&branch=master&keyset={keyset_id}'
TANKER_KEY_URL = 'http://tanker.yandex-team.ru/?project={project}&branch=master&keyset={keyset_id}&key={key_id}'
TANKER_EXPORT_URL = ('https://tanker-api.yandex-team.ru/keysets/tjson/?'
                     'project-id={project}&branch-id=master&keyset-id={keyset_id}')
BULK_SIZE = 100


def is_empty_string(s):
    """
    some translations in Yandex.Tanker contains garbage beginning and trailing space symbols which we cut here
    """
    return not bool(re_strip.sub(ur'\1', s.strip()))


class TranslationsUpdater(object):
    def __init__(self):
        self.localized_items = []

    def update(self, translation_ids):
        """
        :param translation_ids: dict - {'project': 'launcher', 'keyset-id': 'themes', 'statuses':['approved']}
        :return: dict object (json parsed file) returned from the Yandex.Tanker
        """
        logger.info('Importing translations.')
        for keyset_info in translation_ids:
            use_statuses = keyset_info['statuses']
            data = self.download_translations(keyset_info['project'], keyset_info['keyset_id'])
            for keyset in data['keysets'].itervalues():
                for key, translation in keyset['keys'].iteritems():
                    localization = {"_id": key, "values": []}
                    translation_list = translation['translations']
                    languages = frozenset(key.lower() for key in translation_list)
                    added_locales = set()
                    for locale, translation_data in translation_list.iteritems():
                        country = None
                        language = locale[:2].lower()
                        if len(locale) == 5:  # like "pt-br"
                            country = locale[3:].upper()
                        elif len(locale) > 2:  # like "zh-Hant"
                            if language in languages and translation_list[language].get('status') in use_statuses:
                                continue  # Skip in favor of the general value
                        if (language, country) in added_locales:
                            continue
                        if translation_data.get('status') in use_statuses:
                            value = {
                                "conditions": {
                                    "locale": {
                                        "language": language
                                    }
                                },
                                "value": translation_data['form']
                            }
                            if country:
                                value['conditions']['locale']['country'] = country
                            localization['values'].append(value)
                            added_locales.add((language, country,))
                        elif not is_empty_string(translation_data['form']):
                            logger.warning('Translation exists, but status is not in %r: '
                                           'lang(%s), key(%s), status(%s), form(%s)',
                                           use_statuses, language, key, translation_data.get('status'),
                                           translation_data['form'].strip())
                    if len(localization['values']) > 0:
                        self.localized_items.append(localization)

    def commit(self):
        logger.info('Updating translations.')
        if self.localized_items:
            db_collection = localization_mongo.db[app.config['LAUNCHER_TRANSLATIONS_COLLECTION']]
            bulk = db_collection.initialize_unordered_bulk_op()
            for i, item in enumerate(self.localized_items, 1):
                bulk.find({'_id': item['_id']}).upsert().update_one(
                    {'$set': {
                        'values': item['values'],
                        'updated_at': datetime.utcnow(),
                    }})
                if i % BULK_SIZE == 0:
                    bulk.execute()
                    bulk = db_collection.initialize_unordered_bulk_op()
            if i % BULK_SIZE:
                bulk.execute()
        logger.info('Translations has been updated.')

    @staticmethod
    def download_translations(project, keyset_id):
        """
        :param project: project in Yandex.Tanker
        :param keyset_id: keyset_id in Yandex.Tanker
        :return: dict object (json parsed file) returned from the Yandex.Tanker
        """
        url = TANKER_EXPORT_URL.format(project=project, keyset_id=keyset_id)
        logger.info('Downloading url=%s', url)
        headers = {'AUTHORIZATION': 'OAuth {token}'.format(token=app.config['TANKER_OAUTH_TOKEN'])}
        response = requests.get(url=url, headers=headers, timeout=10)
        response.raise_for_status()
        translations = response.json()
        logger.info('Downloading done.')
        return translations
