from hashlib import md5

from flask import url_for, current_app as app, request, flash, redirect, Markup, abort
from flask.views import MethodView
from flask_admin import expose
from flask_admin.babel import gettext
from flask_admin.base import AdminIndexView, BaseView
from flask_admin.contrib.mongoengine import ModelView as BaseModelView
from flask_admin.form import FormOpts
from flask_admin.helpers import get_redirect_target
from flask_admin.model.helpers import get_mdict_item_or_list
from flask_admin.model.template import BaseListRowAction
from flask_login import current_user, login_user
from six import string_types
from werkzeug.exceptions import NotFound

from jafar import yandex_oauth, app_info_loader
from jafar.admin.fields import PrettyJSONField, S3ImageUploadField
from jafar.admin.models.base import ROLE_ADMIN, ROLE_DEVELOPER, ROLE_SUPPORT, History, AdminUser
from jafar.web import JsonResponse

DEV_USERNAME = 'dev'


class AuthMixin(object):
    access_roles = ROLE_ADMIN

    def is_accessible(self):
        return app.debug or (
                current_user.is_authenticated and
                any(current_user.has_role(role) for role in self.access_roles)
        )

    @staticmethod
    def inaccessible_callback(name, **kwargs):
        return yandex_oauth.authorize(callback=url_for('oauth_callback', _external=True, _scheme='https'))


class JafarAdminIndexView(AuthMixin, AdminIndexView):
    @staticmethod
    def is_accessible():
        """ All authenticated users can view the main page """
        return app.debug or current_user.is_authenticated

    @expose('/app')
    def app_info(self, *args, **kwargs):
        package_name = request.args['package_name']
        apps_info = app_info_loader.load(
            package_names=[package_name],
            language='ru',
            raise_on_missing=False
        )
        if package_name not in apps_info:
            return NotFound('App not found')
        return self.render('app.html', app=apps_info[package_name])


class AuthAdminView(AuthMixin, BaseView):
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER


class ModelView(AuthMixin, BaseModelView):
    access_roles = ROLE_ADMIN, ROLE_DEVELOPER

    column_exclude_list = ('_cls',)
    list_template = 'list.html'
    edit_template = 'edit.html'
    create_template = 'create.html'
    copy_template = 'copy.html'

    @property
    def can_edit(self):
        return self.is_accessible()

    @property
    def can_delete(self):
        return self.can_edit and not (
                current_user.is_authenticated and current_user.has_role(ROLE_SUPPORT)
        )

    can_copy = can_create = can_edit

    def update_model(self, form, model):
        old_model = model.to_json()
        result = super(ModelView, self).update_model(form, model)
        new_model = model.to_json()
        self.save_to_history('update', model, before=old_model, after=new_model)
        return result

    def create_model(self, form):
        model = super(ModelView, self).create_model(form)
        if model:
            self.save_to_history('create', model, after=model.to_json())
        return model

    def delete_model(self, model):
        self.save_to_history('delete', model, before=model.to_json())
        return super(ModelView, self).delete_model(model)

    def save_to_history(self, action, model, before='', after=''):
        History(DEV_USERNAME if current_user.is_anonymous else current_user.username,
                action=action,
                document_class=model.__class__.__name__,
                document_before=before,
                document_after=after,
                ).save()

    @expose('/copy/', methods=['GET', 'POST'])
    def copy_view(self):
        """
        Copy view.
        Acts like create view, but loads model copy into the form fields.
        Code is almost the copy of .create_view method.
        Do not forget to update this method when upgrading flask_admin.
        """
        return_url = get_redirect_target() or self.get_url('.index_view')

        if not self.can_create:
            return redirect(return_url)

        model_id = get_mdict_item_or_list(request.args, 'id')
        if model_id is None:
            return redirect(return_url)
        base_model = self.get_one(model_id)

        if base_model is None:
            flash(gettext('Record does not exist.'))
            return redirect(return_url)

        form = self.create_form(obj=base_model)
        if not hasattr(form, '_validated_ruleset') or not form._validated_ruleset:
            self._validate_form_instance(ruleset=self._form_create_rules, form=form)

        if self.validate_form(form):
            model = self.create_model(form)
            if model:
                flash(gettext('Record was successfully created.'))
                if '_add_another' in request.form:
                    return redirect(request.url)
                elif '_continue_editing' in request.form:
                    # if we have a valid model, try to go to the edit view
                    if model is not True:
                        url = self.get_url('.edit_view', id=self.get_pk_value(model), url=return_url)
                    else:
                        url = return_url
                    return redirect(url)
                else:
                    # save button
                    return redirect(self.get_save_return_url(model, is_created=True))

        form_opts = FormOpts(widget_args=self.form_widget_args,
                             form_rules=self._form_create_rules)

        return self.render(self.copy_template,
                           form=form,
                           form_opts=form_opts,
                           return_url=return_url)

    @expose('/ajax/lookup/')
    def ajax_lookup(self):
        """ Custom ajax_lookup view with ability to pass additional fields to Ajax lookuper """
        name = request.args.get('name')
        query = request.args.get('query')
        offset = request.args.get('offset', type=int)
        limit = request.args.get('limit', 10, type=int)
        loader = self._form_ajax_refs.get(name)

        if not loader:
            abort(404)

        additional_fields = getattr(loader, 'additional_fields', [])
        kwargs = {field: request.args.get(field) for field in additional_fields}
        data = [loader.format(m) for m in loader.get_list(query, offset, limit, **kwargs)]
        return JsonResponse(data)


class OAuthCallback(MethodView):
    """ Handles getting token from yandex-team OAuth and user info processing """

    def get(self):
        if current_user.is_anonymous:
            user = self.get_oauth_user()
            login_user(user, remember=True)
        return redirect(url_for('admin.index'))

    @staticmethod
    def get_oauth_user():
        resp = yandex_oauth.authorized_response()
        token = resp['access_token']

        user_info = yandex_oauth.get('https://login.yandex-team.ru/info', token=[token]).data
        email = user_info['default_email']
        username = user_info['login']
        return AdminUser.objects(email=email).first() or AdminUser(email=email, username=username).save()


class OnlyOneDocumentView(ModelView):
    """ View for capped collections with one document (max_documents: 1) """
    template = 'edit_only.html'

    @expose('/', methods=['GET', 'POST'])
    def index_view(self):
        """
        Edits the only document in capped collections
        """
        return_url = get_redirect_target() or self.get_url('admin.index')

        if not self.can_edit:
            return redirect(return_url)

        form = self.create_form(obj=self.model.objects.first())
        if not hasattr(form, '_validated_ruleset') or not form._validated_ruleset:
            self._validate_form_instance(ruleset=self._form_create_rules, form=form)

        if self.validate_form(form):
            model = self.create_model(form)
            if model:
                flash(gettext('Record was successfully saved.'))
                return redirect(return_url)

        form_opts = FormOpts(widget_args=self.form_widget_args, form_rules=self._form_create_rules)
        return self.render(self.template, form=form, form_opts=form_opts, return_url=return_url)


class GlobalConfigView(OnlyOneDocumentView):
    form_overrides = {
        'placement_mapping': PrettyJSONField,
    }


class IconLinkRowAction(BaseListRowAction):
    template = (u'<a href="{url}" title="{title}">'
                u'<img src="{icon_url}"  style="width:14px;margin:-4px 0px 0px 4px;">'
                u'</a>')

    def __init__(self, icon_url, action_url, title=None):
        super(IconLinkRowAction, self).__init__(title=title)
        self.action_url = action_url
        self.icon_url = icon_url

    def render(self, context, row_id, row):
        if isinstance(self.action_url, string_types):
            url = self.action_url.format(row_id=row_id)
        else:
            url = self.action_url(self, row_id, row)
        return Markup(self.template.format(url=url, icon_url=self.icon_url, title=self.title))


def get_md5_image_namegen(prefix):
    """ Generates unique name for file based on its md5 """

    # noinspection PyUnusedLocal
    def namegen(model, data):
        md5_hex = md5(data.read()).hexdigest()
        file_name, _, extension = data.filename.rpartition('.')
        data.stream.seek(0)
        return '%s/%s_%s.%s' % (prefix, file_name, md5_hex, extension)

    return namegen


def get_s3_image_subdocument_config(prefix):
    return {
        'form_columns': ('key',),
        'form_extra_fields': {
            'key': S3ImageUploadField(namegen=get_md5_image_namegen(prefix), label='File'),
        }
    }
